package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.awt.Dimension;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import org.apache.commons.lang3.ObjectUtils;

import net.sf.gluebooster.java.booster.basic.math.ConditionBasics;
import net.sf.gluebooster.java.booster.essentials.math.Condition;
import net.sf.gluebooster.java.booster.essentials.meta.Codable;
import net.sf.gluebooster.java.booster.essentials.utils.Check;
import net.sf.gluebooster.java.booster.essentials.utils.MathBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.ReflectionBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;

/**
 * Abstract properties of elements that may be positioned on a table or that
 * describe places on the table.
 * 
 * @author CBauer
 * 
 * @param <Implementation>
 */
public abstract class AbstractTableElement<Implementation extends AbstractTableElement<Implementation>>
		extends ConditionBasics<Implementation> implements TableElement<Implementation>, 
		Codable {

	/**
	 * The x-position of this element
	 * 
	 */
	private Integer x;
	/**
	 * The x-position of this element
	 * 
	 */
	private Integer y;

	public Integer getX() {
		return x;
	}

	public void setX(Integer x) {
		this.x = x;
	}

	public Integer getY() {
		return y;
	}

	public void setY(Integer y) {
		this.y = y;
	}

	/**
	 * Throws an exception if any position value is null.
	 * 
	 * @return the position of this element.
	 */
	public Dimension getPosition() {
		return new Dimension(x, y);
	}

	/**
	 * Adds by updating all fields if possible
	 */
	@Override
	protected void addFrom(Implementation condition) throws Exception {
		ArrayList<Field> fields = new ArrayList<Field>();
		Class theClass = getClass();
		// get all fields defined in subclasses up to this class
		do {
			for (Field field : theClass.getDeclaredFields()) {
				fields.add(field);
			}
			theClass = theClass.getSuperclass();
		} while (!theClass.equals(AbstractTableElement.class));

		// update all values
		for (Field field : fields) {
			String fieldname = field.getName();
			Object thisValue = ReflectionBoostUtils
					.getProperty(this, fieldname);
			Object otherValue = ReflectionBoostUtils.getProperty(condition,
					fieldname);

			if (thisValue == null) {
				ReflectionBoostUtils.setProperty(this, fieldname, otherValue);
			} else {
				// check for contradictions
				if (otherValue != null) {
					Check.isTrue(thisValue.equals(otherValue),
							"contradicting values field ", fieldname,
							" thisValue: ", thisValue, " otherValue: ",
							otherValue, " this: ", this, " other: ", condition);
				}
			}
		}


	}

	@Override
	public <Result extends Condition> Result findSubcondition(Result template)
			throws Exception {
		throw new UnsupportedOperationException(
				"this method must be overwritten by subclasses");
	}

	/**
	 * Is this element in conflict with a given condition.
	 * 
	 * @param condition
	 *            the condition to be tested.
	 * @param defaultValue
	 *            the default value that is returned if the condition can not be
	 *            handled.
	 * @return true if conflicting
	 */
	protected Boolean isInConflict(Condition condition, Boolean defaultValue)
			throws Exception {
		boolean unknown = false;
		if (condition instanceof Collection) {
			for (Condition cond : (Collection<Condition>) condition) {
				Collection conflictResult = isInConflict(condition);
				if (conflictResult == null) {
					unknown = true;
				} else {
					return conflictResult.isEmpty();
				}
			}
			if (unknown)
				return null;
			else
				return Boolean.FALSE;

		} else {
			return defaultValue;
		}
	}

	/**
	 * Could one object be another Null is a wildcard
	 * 
	 * @param object1
	 *            the first object
	 * @param object2
	 *            can the first object be this object
	 * @param neutralElement
	 *            the element that could be anything. Null additionally is always a neutral element
	 * @return true if object1 can be object2
	 */
	protected boolean couldBe(Object object1, Object object2,
			Object neutralElement) {
		return (object1 == null) || (object2 == null)
				|| ObjectUtils.equals(object1, neutralElement)
				|| ObjectUtils.equals(object2, neutralElement)
				|| object1.equals(object2);
	}

	/**
	 * Can this element be another element. Example a quadruple block at a given
	 * position can be another quadruple block with an unknown position.
	 * 
	 * @param element
	 *            the compared element
	 * @return true, if this element can be the compared element.
	 */
	protected boolean couldBe(Implementation element) {
		if (!getId().equals(element.getId()))
			return false;

		if (!couldBe(getName(), element.getName(), null))
			return false;
		if (!couldBe(getX(), element.getX(), null))
			return false;
		if (!couldBe(getY(), element.getY(), null))
			return false;

		return true;
	}

	@Override
	protected Collection impliesImplementation(Implementation condition2)
			throws Exception {

		if (couldBe(condition2)) {
			return Collections.EMPTY_LIST;
		} else {
			return Arrays.asList("this could not be condition2");
		}
	}

	@Override
	public void checkThatAllStatesAreKnown() throws IllegalStateException {
		Check.notNull(x, "x");
		Check.notNull(y, "y");
	}


	@Override
	public String asSourcecode(StringBuilder result, Object... parameters)
			throws Exception {
		String uid = MathBoostUtils.getTemporaryId();

		String classname = getClass().getSimpleName();
		String objectname = "the" + classname + uid;
		TextBoostUtils.append(result, "\n", classname, " ", objectname,
				"= new ", classname, "();\n");
		if (x != null) {
			TextBoostUtils.append(result, objectname, ".setX(", x, ");\n");
		}

		if (y != null) {
			TextBoostUtils.append(result, objectname, ".setY(", y, ");\n");
		}

		TextBoostUtils.append(result, objectname, ".setId(\"", "" + getId(),
				"\");\n");
		TextBoostUtils.append(result, objectname, ".setName(\"", getName(),
				"\");\n");

		return objectname;
	}

	@Override
	public String toString(){
		StringBuilder result = new StringBuilder();
		result.append(getName());

		if (x != null || y != null){
			result.append("(").append(getX()).append(", ").append(getY())
				.append(") ");
		}
		
		return result.toString();
	}

	/**
	 * Sets the position of this element.
	 * 
	 * @return this
	 */
	public Implementation setPosition(Integer x, Integer y) throws Exception {
		setX(x);
		setY(y);
		return (Implementation) this;
	}

	/**
	 * Sets the position of this element.
	 * 
	 * @return this
	 */
	public Implementation setPosition(Dimension position) throws Exception {

		return setPosition(position.width, position.height);
	}


}
