package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.sf.gluebooster.java.booster.essentials.math.Condition;
import net.sf.gluebooster.java.booster.essentials.utils.Check;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;

/**
 * A block of the blockworld. The name is a String
 * 
 * @defaultParamText name the name of this block
 * @defaultParamText x the x-position on the table
 * @defaultParamText y the y-position on the table
 * 
 * @author CBauer
 * 
 */
public class Block extends AbstractTableElement<Block> {


	/**
	 * The type of the block.
	 */
	private Blocktype type;
	/**
	 * The orientation of the block
	 */
	private Orientation orientation;

	/**
	 * Is the block lying (on another block) on the table or in the air (and
	 * held by a hand).
	 */
	private Boolean up;
	/**
	 * The names of the blocks that directly lie under this block A null value
	 * means that nothing is known.
	 */
	private Set<Object> overBlocks;

	public Block() {
	}

	/**
	 * Constructor with name.
	 */
	public Block(Object name) {
		setName(name);
	}


	public Block(Object name, Integer x, Integer y, Blocktype type,
			Orientation orientation, Boolean up) {
		this(name);
		setX(x);
		setY(y);
		this.type = type;
		this.orientation = orientation;
		this.up = up;
	}

	public Block(Object name, Integer x, Integer y, Blocktype type,
			Orientation orientation, Boolean up, Set<Object> overBlocks) {
		this(name, x, y, type, orientation, up);
		this.overBlocks = overBlocks;
	}

	/**
	 * 
	 * @param overBlock
	 *            the name of the block this block is lying on.
	 */
	public Block(Object name, Integer x, Integer y, Blocktype type,
			Orientation orientation, Boolean up, String overBlock) {
		this(name, x, y, type, orientation, up);
		this.overBlocks = new HashSet<Object>();
		if (overBlock != null)
			overBlocks.add(overBlock);
	}


	public Blocktype getType() {
		return type;
	}

	public void setType(Blocktype type) {
		this.type = type;
	}

	public Orientation getOrientation() {
		return orientation;
	}

	public void setOrientation(Orientation orientation) {
		this.orientation = orientation;
	}

	public Block setOrientationReturnThis(Orientation orientation) {
		setOrientation(orientation);
		return this;
	}

	/**
	 * Sets the position and orientation of this block
	 * 
	 * @return this
	 */
	public Block setPosition(Integer x, Integer y, Orientation orientation)
			throws Exception {
		setPosition(x, y);
		setOrientation(orientation);
		return this;
	}

	public Boolean getUp() {
		return up;
	}

	public void setUp(Boolean up) {
		this.up = up;
	}

	@Override
	public String toString() {

		StringBuilder result = new StringBuilder(super.toString());

		addIfNotNull(null, orientation, " ", result);
		addIfNotNull(null, type, " ", result);

		if (Boolean.TRUE.equals(up)) {
			result.append("up ");
		} else if (Boolean.FALSE.equals(up)) {
			result.append("down ");
			if (overBlocks != null && !overBlocks.isEmpty()) {
				result.append(" over blocks ").append(overBlocks);
			}

		}

		return result.toString();
	}

	public Set<Object> getOverBlocks() {
		return overBlocks;
	}

	public void setOverBlocks(Set<Object> overBlocks) {
		this.overBlocks = overBlocks;
	}



	/**
	 * TODO refactor into BoostedObject
	 * 
	 * @return
	 * @throws CloneNotSupportedException
	 */
	@Override
	public Block cloneMe() throws Exception {
		Block result = super.cloneMe();
		if (overBlocks != null) {
			result.overBlocks = new HashSet<Object>(overBlocks);
		}
		return result;
	}

	public Boolean isOverBlocks() {
		if (overBlocks == null) {
			return null;
			// do not know
		} else {
			return !overBlocks.isEmpty();
		}
	}

	/**
	 * Is this block over another block.
	 * 
	 * @param nameOfBlock
	 *            the name of the other block.
	 * @return null if not enough is known
	 */
	public Boolean isOverBlock(String nameOfBlock) {
		if (overBlocks == null) {
			return null;
			// do not know
		} else {
			return overBlocks.contains(nameOfBlock);
		}
	}

	/**
	 * Get all pairs (x/y) where the block lies. If the block is up, an empty
	 * collection is returned.
	 * 
	 * @return Dimension x=width, y = height
	 */
	public Collection<Place> getOccupiedPlaces() {
		ArrayList<Place> result = new ArrayList<Place>();

		if (Boolean.FALSE.equals(up) && getX() != null && getY() != null) {
			result.add(new Place(getPosition()));
			if ((!Blocktype.SINGLEBLOCK.equals(type))
					&& !Orientation.Z.equals(orientation)) {
				if (Orientation.X.equals(orientation)) {
					result.add(new Place(getX() + 1, getY()));
					if (Blocktype.QUADRUPLEBLOCK.equals(type)) {
						result.add(new Place(getX() + 2, getY()));
						result.add(new Place(getX() + 3, getY()));
					}

				} else { // Y
					result.add(new Place(getX(), getY() + 1));
					if (Blocktype.QUADRUPLEBLOCK.equals(type)) {
						result.add(new Place(getX(), getY() + 2));
						result.add(new Place(getX(), getY() + 3));
					}

				}
			}
		}

		return result;
	}

	/**
	 * Does this block occupy a given place.
	 * 
	 * @param place
	 *            the place to be tested
	 * @return true if the place is occupied by this block.
	 */
	public boolean occupies(Place place) {
		for (Place position : getOccupiedPlaces()) {
			if (position.getX().equals(place.getX())
					&& position.getY().equals(place.getY()))
				return true;
		}

		return false;
	}

	/**
	 * Does this block occupy a given place.
	 * 
	 * @param places
	 *            the places that are tested
	 * @return true if the block occupies at least on place
	 */
	public boolean occupies(Collection<? extends Place> places) {
		for (Place place : places) {
			if (occupies(place))
				return true;
		}

		return false;
	}

	/**
	 * TODO refactor into boostedObject, so that the names work as ids.
	 */
	@Override
	public void setName(Object name) {
		super.setName(name);
		super.setId(name);
	}

	@Override
	public Collection isInConflict(Condition condition) throws Exception {
		if (condition instanceof Block) {
			Block block = (Block) condition;
			if (!couldBe(block)) {
				return Arrays.asList("this could not be block");
			} else if (!block.couldBe(this)) {
				return Arrays.asList("block could not be this");
			} else {
				return Collections.EMPTY_LIST;
			}

		} else {
			Boolean result = isInConflict(condition, Boolean.FALSE);
			if (result == null) {
				return null;
			} else if (result.booleanValue()) {
				return Arrays.asList("a conflict exists");
			} else {
				return Collections.EMPTY_LIST;
			}
		}
	}

	@Override
	protected boolean couldBe(Block block) {
		if (!super.couldBe(block))
			return false;

		if (!couldBe(getUp(), block.getUp(), null))
			return false;
		if (!couldBe(getType(), block.getType(), Blocktype.ANYBLOCK))
			return false;
		if (!couldBe(getOrientation(), block.getOrientation(), Orientation.ANY))
			return false;
		if (!couldBe(getOverBlocks(), block.getOverBlocks(),
				Collections.EMPTY_SET))
			return false;

		return true;

	}

	@Override
	public void checkThatAllStatesAreKnown() throws IllegalStateException {
		Check.notNull(orientation, "orientation");
		Check.notNull(type, "type");
		Check.notNull(up, "up");
	}

	@Override
	public String asSourcecode(StringBuilder result, Object... parameters)
			throws Exception {
		String name = super.asSourcecode(result, parameters);
		if (orientation != null) {
			TextBoostUtils.append(result, name, ".setOrientation(Orientation.",
				orientation, ");\n");
		}

		if (type != null) {
			TextBoostUtils.append(result, name, ".setType(Blocktype.", type,
					");\n");
		}

		if (up != null) {
			TextBoostUtils.append(result, name, ".setUp(", up, ");\n");
		}

		if (overBlocks != null) {
			TextBoostUtils.append(result, name,
					".setOverBlocks(new HashSet<String>());\n");
			for (Object blockName : overBlocks) {
				TextBoostUtils.append(result, name, ".getOverBlocks().add(\"",
						blockName, "\");\n");
			}
		}
		return name;
	}

	@Override
	public Block cloneMeOnlyName() throws Exception {

		Block result = super.cloneMeOnlyName();
		result.setType(type);
		return result;
	}

	/**
	 * Returns the coordinates that are occupied by this block.
	 * 
	 * The coordinates are computed, even if the block is up. (Then the z-index
	 * should indicate that the block is up).
	 * 
	 * @param z
	 *            the bottommost z-position
	 * @return list of [x,y,z]
	 */
	public List<int[]> getOccupiedPoints(int z) {
		ArrayList<int[]> result = new ArrayList<int[]>();

			result.add(new int[] { getX(), getY(), z });

			if (!Blocktype.SINGLEBLOCK.equals(type)) {
				if (Orientation.X.equals(orientation)) {
					result.add(new int[] { getX() + 1, getY(), z });
				}
				if (Orientation.Y.equals(orientation)) {
					result.add(new int[] { getX(), getY() + 1, z });
				}
				if (Orientation.Z.equals(orientation)) {
					result.add(new int[] { getX(), getY(), z + 1 });
				}

				if (Blocktype.QUADRUPLEBLOCK.equals(type)) {
					if (Orientation.X.equals(orientation)) {
						result.add(new int[] { getX() + 2, getY(), z });
						result.add(new int[] { getX() + 3, getY(), z });
					}
					if (Orientation.Y.equals(orientation)) {
						result.add(new int[] { getX(), getY() + 2, z });
						result.add(new int[] { getX(), getY() + 3, z });
					}
					if (Orientation.Z.equals(orientation)) {
						result.add(new int[] { getX(), getY(), z + 2 });
						result.add(new int[] { getX(), getY(), z + 3 });
					}
				}
			}

		return result;
	}
}
