package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import net.sf.gluebooster.java.booster.basic.container.BoostedNode;
import net.sf.gluebooster.java.booster.basic.math.Operation;
import net.sf.gluebooster.java.booster.basic.math.Operator;
import net.sf.gluebooster.java.booster.basic.math.graph.BoostedNodeGraph;
import net.sf.gluebooster.java.booster.basic.math.planning.Plan;
import net.sf.gluebooster.java.booster.basic.math.planning.World;
import net.sf.gluebooster.java.booster.basic.meta.DocumentationContext;
import net.sf.gluebooster.java.booster.basic.text.general.BookBoostUtils;
import net.sf.gluebooster.java.booster.essentials.eventsCommands.Callable;
import net.sf.gluebooster.java.booster.essentials.utils.Check;

/**
 * {@Feature}Blockworld, a world full of blocks to move.{@FeatureEnd} A description of the properties and operations of the blockworld.
 * 
 * @author CBauer
 * 
 */
public class Blockworld extends World<Table> {

	// static final Operator OPERATOR_NOTHING_TO_DO = new Operator(
	// OPERATORNAME_NOTHING_TO_DO, 0);

	/**
	 * The node is still to solve.
	 */
	static final Operator OPERATOR_TO_SOLVE = new Operator(
			OPERATORNAME_TO_SOLVE, 10);
	// private static final Operator OPERATOR_UNDEFINED = new Operator(
	// OPERATORNAME_UNDEFINED, 0);

	/**
	 * Move up a hand
	 */
	static final Operator OPERATOR_MOVE_UP = new Operator(
			"MOVE_UP", 1, true);

	/**
	 * Move down a hand
	 */
	static final Operator OPERATOR_MOVE_DOWN = new Operator(
"MOVE_DOWN", 1,
			true);
	/**
	 * Move a hand (the hand must be up)
	 */
	static final Operator OPERATOR_MOVE = new Operator("MOVE",
			1, true);

	/**
	 * Turn a hand
	 */
	static final Operator OPERATOR_TURN = new Operator("TURN",
			1, true);

	/**
	 * Grab (a block) with a hand
	 */
	static final Operator OPERATOR_GRAB = new Operator("GRAB",
			1, true);
	/**
	 * Release (a block) with a hand
	 */
	static final Operator OPERATOR_RELEASE = new Operator(
			"RELEASE", 1, true);

	/**
	 * Release the block that is carried by a hand. Move it somewhere and relase
	 */
	static final Operator OPERATOR_RELEASE_BLOCK = new Operator(
			"RELEASE_BLOCK", 3);

	/**
	 * Grab a block somewhwere on the table with a hand
	 */
	static final Operator OPERATOR_GRAB_BLOCK = new Operator(
			"GRAB_BLOCK", 3);

	/**
	 * Move the hand to a place on the table
	 */
	static final Operator OPERATOR_MOVE_HAND = new Operator(
			"MOVE_HAND", 2);

	/**
	 * Remove all blocks lying on top of a given block, so that the block lies
	 * free.
	 */
	static final Operator OPERATOR_FREE_THE_BLOCK = new Operator(
			"FREE_THE_BLOCK", 2);
	/**
	 * Move a block to a place on the table
	 */
	static final Operator OPERATOR_MOVE_BLOCK = new Operator(
			"MOVE_BLOCK", 9);

	/**
	 * Remove a block from its position and put it elsewhere
	 */
	static final Operator OPERATOR_REMOVE_BLOCK = new Operator(
			"REMOVE_BLOCK", 3);
	/**
	 * Clear all blocks lying on given places, so that the surface of the table
	 * is visible/accessible.
	 */
	static final Operator OPERATOR_MAKE_SURFACE = new Operator(
			"MAKE_SURFACE", 4);

	/**
	 * Create a new virtual object.
	 */
	static final Operator OPERATOR_NEW_VIRTUAL_OBJECT = new Operator(
			OPERATORNAME_NEW_VIRTUAL_OBJECT, 0, true);

	/**
	 * Delete a virtual object.
	 */
	static final Operator OPERATOR_DELETE_VIRTUAL_OBJECT = new Operator(
			OPERATORNAME_DELETE_VIRTUAL_OBJECT, 0,
			true);

	// Names of used properties
	/**
	 * The property that some blocks are moved
	 */
	private static final String PROPERTY_MOVED_BLOCKS = "MOVED BLOCKS: ";
	/**
	 * The property that some blocks are freed
	 */
	private static final String PROPERTY_NEW_FREE_BLOCKS = "NEW FREE BLOCKS: ";
	/**
	 * The property that some blocks are covered.
	 */
	private static final String PROPERTY_NEW_NON_FREE_BLOCKS = "NEW NON-FREE BLOCKS: ";

	/**
	 * The property that some hands did something.
	 */
	private static final String PROPERTY_CHANGED_HANDS = "CHANGED HANDS : ";
	/**
	 * The property that a hand is a virtual hand and not yet fixed.
	 */
	private static final String PROPERTY_VIRTUAL_HAND = "VIRTUAL HAND:";

	// #define MOVED_BLOCKS_STRING "MOVED BLOCKS : "
	// #define NEW_FREE_BLOCKS_STRING "NEW FREE BLOCKS : "
	// #define NEW_NON_FREE_BLOCKS_STRING "NEW NON-FREE BLOCKS : "
	// #define CHANGED_HANDS_STRING "CHANGED HANDS : "
	// #define NEW_TEMPORARY_HAND_STRING "NEW TEMPORARY HAND : "
	// #define DELETE_TEMPORARY_HAND_STRING "NEW TEMPORARY HAND : "

	/**
	 * All operators of the blockworld.
	 */
	private static List<Operator> ALL_OPERATORS = Arrays.asList(
			OPERATOR_TO_SOLVE, /* OPERATOR_UNDEFINED, */OPERATOR_MOVE_UP,
			OPERATOR_MOVE_DOWN, OPERATOR_MOVE, OPERATOR_TURN, OPERATOR_GRAB,
			OPERATOR_RELEASE, OPERATOR_RELEASE_BLOCK, OPERATOR_GRAB_BLOCK,
			OPERATOR_MOVE_HAND, OPERATOR_FREE_THE_BLOCK, OPERATOR_MOVE_BLOCK,
			OPERATOR_REMOVE_BLOCK, OPERATOR_MAKE_SURFACE,
			OPERATOR_NEW_VIRTUAL_OBJECT, OPERATOR_DELETE_VIRTUAL_OBJECT);

	/**
	 * Expanders expand nodes with operations that are not elementary.
	 */
	private static Map<Operator, Collection<Callable<Object/* BoostedNode */, BoostedNodeGraph>>> expanders = createExpanders();

	/**
	 * Creator of virtual objects.
	 */
	private static BlockworldVirtualObjectInstantiator instantiator = new BlockworldVirtualObjectInstantiator();

	public Blockworld() {
		super(getPossibleConflictingPropertynames());
	}

	/**
	 * Tell the names of the properties which can be in conflict.
	 * 
	 * @return a collection of two property names. The two properties may be in conflict with each other (An object with property 1 may be in conflict with
	 *         another object with property 2).
	 */
	private static Collection<Pair<String, String>> getPossibleConflictingPropertynames() {
		ArrayList<Pair<String, String>> result = new ArrayList<Pair<String, String>>();
		result.add(new ImmutablePair<String, String>(PROPERTY_MOVED_BLOCKS,
				PROPERTY_MOVED_BLOCKS));// moving the same block in parallel
										// branches may cause problems
		result.add(new ImmutablePair<String, String>(PROPERTY_MOVED_BLOCKS,
				PROPERTY_NEW_FREE_BLOCKS));// moving a block in parallel to
											// uncovering it may cause problems
		result.add(new ImmutablePair<String, String>(PROPERTY_MOVED_BLOCKS,
				PROPERTY_NEW_NON_FREE_BLOCKS));// moving a block in parallel to
												// covering it may cause
												// problems
		result.add(new ImmutablePair<String, String>(PROPERTY_CHANGED_HANDS,
				PROPERTY_CHANGED_HANDS));// manipulating the same hand in
											// parallel branches may cause
											// problems.

		return result;
	}

	/**
	 * Create expanders for the different operatos
	 * 
	 * @return the mapping of the operator to the corresponding expander
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private static Map<Operator, Collection<Callable<Object/* BoostedNode */, BoostedNodeGraph>>> createExpanders() {

		HashMap result = new HashMap();
		result.put(OPERATOR_TO_SOLVE,
				Arrays.asList(BlockworldNodeExpander.toSolveExpander));
		result.put(OPERATOR_MOVE_BLOCK,
				Arrays.asList(BlockworldNodeExpander.moveBlockExpander));
		result.put(OPERATOR_MAKE_SURFACE,
				Arrays.asList(BlockworldNodeExpander.makeSurfaceExpander));
		result.put(OPERATOR_GRAB_BLOCK,
				Arrays.asList(BlockworldNodeExpander.grabBlockExpander));
		result.put(OPERATOR_MOVE_HAND,
				Arrays.asList(BlockworldNodeExpander.moveHandExpander));
		result.put(OPERATOR_FREE_THE_BLOCK,
				Arrays.asList(BlockworldNodeExpander.freeTheBlockExpander));
		result.put(OPERATOR_REMOVE_BLOCK,
				Arrays.asList(BlockworldNodeExpander.removeBlockExpander));

		return result;
	}
	
	@Override
	public boolean isNoNeedToSolve(Operation node) throws Exception {

		return node.isNoNeedToSolve();
		/*
		 * Operator operator = node.getOperator();
		 * 
		 * if (OPERATOR_TO_SOLVE.equals(operator)) { return node.isSolved(); }
		 * else { throw new
		 * IllegalStateException("Operator not (yet) supported: " + operator); }
		 */

		// BOOLEAN result = FALSE;
		// OPERATORNAME name = ( (OPERATOR *)element)->name;
		// NATURAL handnr;
		// NATURAL blocknr;
		// BLOCK *block;
		// HAND *hand;
		// NATURAL new_x;
		// NATURAL new_y;
		// BOOLEAN up;
		// ORIENTATIONS orientation;
		// switch (name)
		// {
		// case TO_SOLVE :
		// result = solved();
		// break;
		// case RELEASE_BLOCK : handnr =( (OPERATOR *)element)->n1;
		// hand = find_hand_in_preconditions( handnr);
		// if ( hand)
		// if( hand->is_empty())
		// result = TRUE;
		// if ( (!precondition) && ( ! global_precondition))
		// result = TRUE;
		// break;
		// case GRAB_BLOCK : handnr =( (OPERATOR *)element)->n2;
		// blocknr =( (OPERATOR *)element)->n1;
		// if ( handnr)
		// {
		// hand = find_hand_in_preconditions( handnr);
		// if ( hand)
		// if ( hand->get_blocknr() == blocknr)
		// result = TRUE;
		// }
		// break;
		// case MOVE_HAND :
		// handnr =( (OPERATOR *)element)->n1;
		// new_x =( (OPERATOR *)element)->n2;
		// new_y =( (OPERATOR *)element)->n3;
		// up = ( (OPERATOR *)element)->boolean;
		// hand = find_hand_in_preconditions( handnr);
		// if ( hand)
		// if( (hand->x == new_x) && ( hand->y == new_y) && ( hand->up == up))
		// result = TRUE;
		// break;
		// case FREE_THE_BLOCK :
		// blocknr =( (OPERATOR *)element)->n1;
		// if ( global_precondition)
		// if ( (( TABLE_CONDITION
		// *)global_precondition)->block_is_free(blocknr))
		// result = TRUE;
		// break;
		// case MOVE_BLOCK :
		// blocknr =( (OPERATOR *)element)->n1;
		// new_x =( (OPERATOR *)element)->n2;
		// new_y =( (OPERATOR *)element)->n3;
		// orientation = ( (OPERATOR *)element)->orientation;
		// block = find_block_in_preconditions( blocknr);
		// if ( block)
		// if ( (block->x == new_x) && (block->y == new_y) &&
		// (block->orientation == orientation))
		// result = TRUE;
		// break;
		// case MAKE_SURFACE :
		// break;// not yet implemented
		// }
		// return result;
		// }

	}

	@Override
	public Collection<Operator> getAllOperators() {
		return ALL_OPERATORS;
	}

	@Override
	public Collection<Callable<Object/* BoostedNode */, BoostedNodeGraph>> getWorldSpecificExpander(
			BoostedNode unsolvedNode) {

		Operation operation = extractOperation(unsolvedNode);
		Collection<Callable<Object/* BoostedNode */, BoostedNodeGraph>> result = expanders
				.get(operation.getOperator());
		if (result == null) {
			result = Collections.emptyList();
		}
		if (result.isEmpty() && !operation.isOperatorElementary()) {
			getLog().warn(
					"Blockworld: no worldSpecificExpander found for operator "
							+ operation.getOperator());
		}
		return result;
	}

	/**
	 * Sets the property value. Todo refactor into basic
	 * 
	 * @param properties
	 *            the properties to be changed
	 * @param key
	 *            the key of the property
	 * @param value
	 *            may be null, then nothing is done.
	 */
	private void putPropertyValue(Map<String, Set<String>> properties,
			String key, Object value) {
		if (value != null) {

			Set<String> values = properties.get(key);
			if (values == null) {
				values = new HashSet<String>();
				properties.put(key, values);
			}
			values.add(value.toString());
		}
	}

	/**
	 * Casts an object to another class
	 * 
	 * @param value
	 *            the inspected object
	 * @param clazz
	 *            the wanted class
	 * @return null if not castable
	 */
	private <Result> Result castTo(Object value, Class<Result> clazz) {
		if (value == null)
			return null;
		else if (value.getClass().equals(clazz)) {
			return (Result) value;
		} else {
			return null;
		}

	}

	/**
	 * Determines the name of the block that is hold by a hand
	 * 
	 * @param hand
	 *            the hand to inspect
	 * @param tablesToInspect
	 *            contains the hand. May be null
	 * @return null if nothing is found
	 */
	private Object getNameOfBlockInHand(Hand hand, Table... tablesToInspect) {
		Object result = null;
		if (hand != null) {
			String handname = hand.getName();
			result = hand.getNameOfBlock();
			for (Table table : tablesToInspect) {
				if (result == null && table != null) {
					result = table.getNameOfBlockOfHandIfPossible(handname);
				}
			}
		}
		return result;
	}

	@Override
	public Map<String, Set<String>> getPropertyValues(
			Operation<Table> node) {
		HashMap<String, Set<String>> result = new HashMap<String, Set<String>>();
		// getLog().warn("forbidden places are not yet evaluated");
		Operator operator = node.getOperator();
		Object operatorDetails = node.getOperatorDetails();
		Table precondition = node.getPrecondition();
		Table globalPrecondition = node.getGlobalPrecondition();
		Table globalPostcondition = node.getGlobalPostcondition();

		// the following variable may be null
		Hand hand = castTo(operatorDetails, Hand.class);
		Object nameOfBlockInHand = getNameOfBlockInHand(hand, precondition,
				globalPrecondition);
		Block block = castTo(operatorDetails, Block.class);
		
		// TODO evaluate forbidden places etc. in precondition

		if (OPERATOR_DELETE_VIRTUAL_OBJECT.equals(operator)) {
			// nothing needs to be done
		} else if (OPERATOR_MAKE_SURFACE.equals(operator)) {
			// nothing needs to be done
		} else if (OPERATOR_MOVE_UP.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			if (nameOfBlockInHand != null) {
				putPropertyValue(result, PROPERTY_MOVED_BLOCKS,
						nameOfBlockInHand);
				if (globalPrecondition != null) {
					if (globalPostcondition == null) {
						throw new IllegalStateException(
								"global postcondition should be known after move up");
					}
					for (Object blockBelowName : globalPrecondition.getBlock(
							nameOfBlockInHand).getOverBlocks()) {
						if (globalPostcondition.isBlockFree(blockBelowName)) {
							putPropertyValue(result, PROPERTY_NEW_FREE_BLOCKS,
									blockBelowName);
						}
					}
				}
			}
		} else if (OPERATOR_MOVE_DOWN.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			if (nameOfBlockInHand != null) {
				putPropertyValue(result, PROPERTY_MOVED_BLOCKS,
						nameOfBlockInHand);
				if (globalPostcondition != null) {
					for (Object blockBelowName : globalPostcondition.getBlock(
							nameOfBlockInHand).getOverBlocks()) {
						putPropertyValue(result, PROPERTY_NEW_NON_FREE_BLOCKS,
								blockBelowName);
					}
				}
			}
		} else if (OPERATOR_MOVE.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, nameOfBlockInHand);
		} else if (OPERATOR_TURN.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, nameOfBlockInHand);
		} else if (OPERATOR_GRAB.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
		} else if (OPERATOR_RELEASE.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
		} else if (OPERATOR_RELEASE_BLOCK.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, nameOfBlockInHand);
		} else if (OPERATOR_GRAB_BLOCK.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
		} else if (OPERATOR_MOVE_BLOCK.equals(operator)) {
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, block.getName());
		} else if (OPERATOR_REMOVE_BLOCK.equals(operator)) {
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, block.getName());
		} else if (OPERATOR_MOVE_HAND.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
			putPropertyValue(result, PROPERTY_MOVED_BLOCKS, nameOfBlockInHand);
		} else if (OPERATOR_NEW_VIRTUAL_OBJECT.equals(operator)) {
			putPropertyValue(result, PROPERTY_CHANGED_HANDS, hand.getName());
		} else if (OPERATOR_FREE_THE_BLOCK.equals(operator)) {
			putPropertyValue(result, PROPERTY_NEW_FREE_BLOCKS, block.getName());
			if (globalPrecondition != null) {
				for (Block blockOnTop : globalPrecondition
						.getBlocksDirectlyOverBlocks(Arrays.asList(block))) {
					putPropertyValue(result, PROPERTY_MOVED_BLOCKS,
							blockOnTop.getName());

				}
			}
		} else if (OPERATOR_TO_SOLVE.equals(operator)) {
			// nothing needs to be done
		} else {
			getLog().warn(
					"getPropertyValues: other operators not yet supported: "
							+ operator);
		}
		
		return result;

		
	}


	@Override
	public Set<Callable> getVirtualObjectInitializer(
			BoostedNode nodeWithVirtualObject) {
		HashSet<Callable> result = new HashSet<Callable>();
		result.add(instantiator);
		return result;
	}

	@Override
	public Table getDefaultPostcondition(Plan<Table> plan,
			Operation<Table> node) throws Exception {
		return BlockworldOperation.getDefaultPostcondition(node,
				plan.getWorldInstanceDescription());
	}

	@Override
	public Table getDefaultPrecondition(Plan<Table> plan,
			Operation<Table> operation) throws Exception {
		return BlockworldOperation.getDefaultPrecondition(operation,
				plan.getWorldInstanceDescription());
	}

	@Override
	public Table getDefaultDeletecondition(Plan<Table> plan,
			Operation<Table> node) throws Exception {
		return BlockworldOperation.getDefaultDeletecondition(node,
				plan.getWorldInstanceDescription());
	}

	@Override
	public Table computeGlobalPostcondition(Operation<Table> operation,
			Table globalPrecondition)
			throws Exception {

		if (globalPrecondition == null)
			return null;

		Table result = null;
		Operator operator = operation.getOperator();
		Table postcondition = operation.getPostcondition();

		if (OPERATOR_TO_SOLVE.equals(operator)) {

			if (Boolean.TRUE.equals(globalPrecondition.implies(postcondition))) {
				result = globalPrecondition.cloneMe();
				// because the problem is solved already and nothing is to do
			} else {
				// nothing can be computed yet
			}
		} else if (OPERATOR_GRAB.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			hand = globalPrecondition.getHand(hand.getName());
			Check.notNull(hand, "hand");
			if (!Boolean.FALSE.equals(hand.getUp())) {
				globalPostconditionWarning(
						"can not grab when hand is not down ", hand);
			}else if (Boolean.TRUE.equals(hand.getHasBlock())) {
				globalPostconditionWarning("can not grab when hand has block ",
						hand);
			} else {
				Integer x = Check.notNull(hand.getX(), "hand x");
				Integer y = Check.notNull(hand.getY(), "hand y");
				Collection<Block> blockToGrab = globalPrecondition
						.getTopmostFreeBlocks(Arrays.asList(new Place(x, y)));
				if (blockToGrab.size() != 1) {
					getLog().warn("no free block at x:", x, " y:", y, " in ",
							globalPrecondition);
				} else {
					result = globalPrecondition.cloneMe();
					Hand resultHand = result.getHand(hand.getName());
					resultHand.setHasBlock(true);
					resultHand.setNameOfBlock(blockToGrab.iterator().next()
							.getName());
				}

			}
		} else if (OPERATOR_RELEASE.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			hand = globalPrecondition.getHand(hand.getName());

			if (!Boolean.FALSE.equals(hand.getUp())) {
				getLog().warn("can not release when hand is not down ", hand);
			} else {
				result = globalPrecondition.cloneMe();
				Hand resultHand = result.getHand(hand.getName());
				resultHand.setHasBlock(false);
				resultHand.setNameOfBlock(null);
			}
		} else if (OPERATOR_NEW_VIRTUAL_OBJECT.equals(operator)) {
			TableElement newObject = operation.getOperatorDetails();
			result = globalPrecondition.cloneMe();
			result.replaceTableElement(newObject);
		} else if (OPERATOR_DELETE_VIRTUAL_OBJECT.equals(operator)) {
			TableElement object = operation.getOperatorDetails();
			result = globalPrecondition.cloneMe();
			result.removeTableElement(object);

		} else if (OPERATOR_MOVE.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			Hand preconditionHand = globalPrecondition.getHand(hand.getName());
			if (!Boolean.TRUE.equals(preconditionHand.getUp())) {
				getLog().warn("can not move when hand is not up ",
						preconditionHand);
			} else if (null == preconditionHand.getX()) {
				getLog().warn("can not move when x is null ", hand);
			} else if (null == preconditionHand.getY()) {
				getLog().warn("can not move when y is null ", hand);
			} else {
				result = globalPrecondition.cloneMe();
				Hand conditionHand = result.getHand(hand.getName());
				conditionHand.setX(hand.getX());
				conditionHand.setY(hand.getY());
				if (Boolean.TRUE.equals(conditionHand.getHasBlock())) {
					Block block = result.getBlock(conditionHand
							.getNameOfBlock());
					block.setX(hand.getX());
					block.setY(hand.getY());
				}
				
			}
		} else if (OPERATOR_MOVE_UP.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			result = globalPrecondition.cloneMe();
			Hand conditionHand = result.getHand(hand.getName());
			conditionHand.setUp(true);
			if (Boolean.TRUE.equals(conditionHand.getHasBlock())) {
				Block block = result.getBlock(conditionHand.getNameOfBlock());
				block.setUp(true);
				block.setOverBlocks(Collections.EMPTY_SET);
			}
		} else if (OPERATOR_MOVE_DOWN.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			result = globalPrecondition.cloneMe();
			Hand conditionHand = result.getHand(hand.getName());
			if (Boolean.FALSE.equals( conditionHand.getUp())){
				//nothing is to do, the hand is already down
			} else {
				conditionHand.setUp(false);
				if (Boolean.TRUE.equals(conditionHand.getHasBlock())) {
					Block block = result.getBlock(conditionHand
							.getNameOfBlock());
					block.setUp(false);
					HashSet<Object> overBlockNames = new HashSet<Object>();
					for (Block overBlock: globalPrecondition.getTopmostFreeBlocks(block.getOccupiedPlaces())){
						overBlockNames.add(overBlock.getName());
					}
					block.setOverBlocks(overBlockNames);
				}
			}
		} else if (OPERATOR_TURN.equals(operator)) {
			Hand hand = operation.getOperatorDetails();
			Hand preconditionHand = globalPrecondition.getHand(hand.getName());
			if (!Boolean.TRUE.equals(preconditionHand.getUp())) {
				getLog().warn("can not turn hand when not up", preconditionHand);
			} else {			
				result = globalPrecondition.cloneMe();
				Hand conditionHand = result.getHand(hand.getName());
				conditionHand.setGrip(hand.getGrip());
				if (Boolean.TRUE.equals(conditionHand.getHasBlock())) {
					Block block = result.getBlock(conditionHand
							.getNameOfBlock());
					block.setOrientation(conditionHand.getGrip());
				}
			}
		} else if (!operator.isElementary()) {
			// nothing can be computed (yet)

		} else {
			getLog().warn(
					"global postconditions are not yet adapted. " + operation);
		}

		if (result == null && operation.isNoNeedToSolve()) {
			result = globalPrecondition.cloneMe();
		}

		if (result != null) {
			result.checkThatAllStatesAreKnown();
			result.checkSingleTableElements();
		}

		return result;
		// special_update_global_postcondition


	}

	@Override
	protected Operation<Table> createOperation() {
		return new BlockworldOperation();
	}

	@Override
	protected void fillWorldDescription(BoostedNode worldChapter, DocumentationContext context) throws Exception {
		if (Locale.GERMAN.getLanguage().equals(context.getLocale().getLanguage())) {
			BookBoostUtils.createDiv(worldChapter,
					"Die Blockwelt ist eine dreidimensionale Welt, ein Tisch, auf dem Dinge (zu Beginn nur Blöcke) liegen. Die Dinge können übereinander gestapelt werden. Sie werden von Händen bewegt.");
		} else {
			BookBoostUtils.createDiv(worldChapter,
					"The blockworld is a three-dimensional world (a table) on which things (at the beginning only blocks) lie. The things can be piled. They are moved by hands. ");
		}
	}
}
