package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import net.sf.gluebooster.java.booster.basic.container.BoostedNode;
import net.sf.gluebooster.java.booster.basic.math.Operation;
import net.sf.gluebooster.java.booster.basic.math.graph.BoostedNodeGraph;
import net.sf.gluebooster.java.booster.basic.math.planning.Plan;
import net.sf.gluebooster.java.booster.basic.math.planning.Planning;
import net.sf.gluebooster.java.booster.essentials.eventsCommands.CallableAbstraction;

/**
 * Expands higher level operations into operations of a lower complexity level.
 * 
 * @defaultParamText node the node to be inspected
 * @author CBauer
 * 
 */
public class BlockworldNodeExpander extends
		CallableAbstraction<Object, BoostedNodeGraph> {

	/**
	 * The default x-coordinate of a new hand.
	 */
	private static final int X_OF_NEW_HAND = -1;

	/**
	 * The default x-coordinate of a new hand.
	 */
	private static final int Y_OF_NEW_HAND = -1;

	/**
	 * The Expander for OPERATOR_TO_SOLVE.
	 */
	static final BlockworldNodeExpander toSolveExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_TO_SOLVE);

	/**
	 * The Expander for OPERATOR_MOVE_BLOCK.
	 */
	static final BlockworldNodeExpander moveBlockExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_MOVE_BLOCK);

	/**
	 * The Expander for OPERATOR_MAKE_SURFACE.
	 */
	static final BlockworldNodeExpander makeSurfaceExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_MAKE_SURFACE);

	/**
	 * The Expander for OPERATOR_GRAB_BLOCK.
	 */
	static final BlockworldNodeExpander grabBlockExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_GRAB_BLOCK);

	/**
	 * The Expander for OPERATOR_MOVE_HAND.
	 */
	static final BlockworldNodeExpander moveHandExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_MOVE_HAND);

	/**
	 * The Expander for OPERATOR_FREE_THE_BLOCK.
	 */
	static final BlockworldNodeExpander freeTheBlockExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_FREE_THE_BLOCK);

	/**
	 * The Expander for OPERATOR_REMOVE_BLOCK.
	 */
	static final BlockworldNodeExpander removeBlockExpander = new BlockworldNodeExpander(
			Blockworld.OPERATOR_REMOVE_BLOCK);

	/**
	 * The type of this expander
	 */
	private Object type;

	private BlockworldNodeExpander(Object type) {
		this.type = type;
	}

	/**
	 * Solve a TO_SOLVE-Node with an elementary postcondition TODO this method is not yet implemented
	 * 
	 * @return the solution/expansion of the node
	 * @throws Exception
	 *             at the moment always
	 */
	public BoostedNodeGraph solveElementary(Operation<Table> node)
			throws Exception {
		throw new IllegalStateException("solveElementary not yet implemented ");
	}

	/**
	 * Solve a TO_SOLVE-node with blocks in the postcondition.
	 * 
	 * Algorithm:
	 * 
	 * Create a tree according to the positions of the blocks (if one block is on top of another then first move the lower block to the correct position, then
	 * the upper block) Add all other predicates as separate nodes of the new graph
	 * 
	 * To avoid forgetting the goal a toSolve is added at the end with low priority
	 * 
	 * @return the solution/expansion
	 */
	public BoostedNodeGraph solveWithBlocks(Operation<Table> node)
			throws Exception {

		Table globalPrecondition = node.getGlobalPrecondition();

		// do only something if the world is known enough (a global precondition
		// is set).
		// Otherwise it is too easy to do unnecessary things, create endless
		// loops, etc.
		if (globalPrecondition == null) {
			return null;
		}

		BoostedNodeGraph result = new BoostedNodeGraph();
		Table postcondition = node.getPostcondition();
		Set<Table> splitPostconditions = postcondition.split();
		int nextPriority = node.getSolvingPriority() - 1;
		
		//maps a block to the created nodes. If a sequence is created the first and the last node are used. If only on node ist created, it is both the first and the last node. 
		HashMap<Block, Pair<BoostedNode, BoostedNode>> blockToFirstLastExpandedNodeMapping = new HashMap<Block, Pair<BoostedNode,BoostedNode>>();
		//mapping of the name to the block
		HashMap<String, Block> namedBlocks = new HashMap<String, Block>();
		
		for (Table partPostcondition : splitPostconditions) {
			if (partPostcondition.hasHandPredicate()) {
				result.addNode(Operation.createBoostedNode(BlockworldOperation
						.toSolve(nextPriority, partPostcondition)));
			} else if (partPostcondition.hasBlockPredicate()) {
				Block block = (Block) partPostcondition.getTableElements()
						.iterator().next();
				
				if (Boolean.FALSE.equals(block.getUp())) {
					// if the block is down
					namedBlocks.put(block.getName().toString(), block);

					BlockworldOperation moveBlockOperation = BlockworldOperation
							.moveBlock(nextPriority, block);
					BoostedNode moveBlockNode = Operation
							.createBoostedNode(moveBlockOperation);
					result.addNode(moveBlockNode);
					
					Boolean isOverBlocks = block.isOverBlocks();
					if (isOverBlocks == null) {
						getLog().debug(
								"solveWithBlocks is not successful:  missing information whether a block is over blocks: ",
								block);
						return null;
					} else if (Boolean.TRUE.equals(isOverBlocks)) {
						blockToFirstLastExpandedNodeMapping.put(block,
								new ImmutablePair<BoostedNode, BoostedNode>(
										moveBlockNode, moveBlockNode));
					} else {
						// if the block lies directly on the table then remove
						// blocks that are in the way before moving the block

						Table freePlacesCondition = partPostcondition
								.createEmptyCondition().addFreePlaces(block);

						BoostedNode makeSurface = Operation.createPredecessor(
								moveBlockNode, BlockworldOperation.makeSurface(
										nextPriority, freePlacesCondition));

						moveBlockOperation
								.addToPrecondition(freePlacesCondition
										.cloneMe());

						blockToFirstLastExpandedNodeMapping.put(block,
								new ImmutablePair<BoostedNode, BoostedNode>(
										makeSurface, moveBlockNode));
					}
				} else {
					getLog().warn("TODO: solve if block is up");
				}

			} else {
				throw new IllegalStateException("table element not supported: "
						+ partPostcondition);
			}
			
				
		}

		// create the tree of the block movements
		for (Map.Entry<Block, Pair<BoostedNode, BoostedNode>> block : blockToFirstLastExpandedNodeMapping
				.entrySet()) {
			Block theBlock = block.getKey();
			if (Boolean.TRUE.equals(theBlock.isOverBlocks())) {
				for (Object nameOfOverThisBlock : theBlock.getOverBlocks())
					result.addEdge(
							blockToFirstLastExpandedNodeMapping.get(
									namedBlocks.get(nameOfOverThisBlock))
									.getRight(), block.getValue().getLeft());
			}
		}

		// add the toSolve node to the sink nodes

		ArrayList<BoostedNode> sinks = new ArrayList<BoostedNode>(
				result.getAllSinkVertices());
		// Create a duplicate of the sinks, because the set may be manipulated
		// during the following operations
		BoostedNode toSolve = result.addNode(Operation
				.createBoostedNode(BlockworldOperation.toSolve(
						nextPriority - 20, postcondition.cloneMe())));

		for (BoostedNode sink : sinks) {
			result.addEdge(sink, toSolve);
		}


		return result;
		// TABLE_CONDITION* postcondition = (TABLE_CONDITION *)(
		// object.postcondition);
		// TABLE_CONDITION *condition_dummy= ( TABLE_CONDITION
		// *)net.CONDITION_dummy;
		// TABLE_CONDITION* precondition = (TABLE_CONDITION *)(
		// object.precondition);
		// TABLE_CONDITION *global_precondition =(TABLE_CONDITION *)(
		// object.global_precondition);
		// BLOCKWORLD_PLAN new_net( net.get_max_handnr_at_start(),
		// condition_dummy->deepcopy_());
		// OBJECT *predicate;
		// BLOCKWORLD_PLAN_OBJECT *net_object;
		// NATURALSET already_correct_positioned_blocks;
		// OBJECTPOINTERSET *non_block_predicates = new OBJECTPOINTERSET;
		//
		// postcondition->reset_predicates();
		//
		// NATURAL new_node_nr = postcondition->max_block_nr() +1;
		// while ( postcondition->next_predicate(&predicate))
		// {
		// //sort_predicate( predicate, non_block_predicates,
		// global_precondition, already_correct_positioned_blocks, new_net,
		// condition_dummy, object, new_node_nr );
		// sort_predicate( predicate, non_block_predicates, global_precondition,
		// new_net, condition_dummy, object, new_node_nr );
		// /* sorts the predicates into
		//
		// - non-block-predicates and already correct postioned blocks :
		// non-block_predicates
		// - not yet correct postioned blocks :
		// these predicates are inserted into the new net
		// according their movement-sequence to achieve the postcondition-table
		// the node-nrs of the predicates are the block-nrs
		// if the block is over a correct positioned block,
		// it is stated that a previous node has the nr of the correct postioned
		// block.
		// This previous node doesn´t exist.
		// Its reference has to be deleted later.
		// The new net has no next-references, only previous - references
		// The net-nodes have no preconditions and not 100% correct
		// postconditions
		// */
		// }
		//
		// // insert the non-block-predicates parallel into the new_net
		// TABLE_CONDITION *non_block_condition = NULL;
		//
		// non_block_predicates->reset();
		// while ( non_block_predicates->next( &predicate))
		// {
		// non_block_condition =( TABLE_CONDITION *)
		// net.CONDITION_dummy->deepcopy();
		// non_block_condition->add_predicate(predicate);
		// net_object = BLOCKWORLD_PLAN_OBJECT::create(NULL,NULL,TO_SOLVE,
		// non_block_condition, NULL);
		// new_net.insert(net_object, NATURALSET::empty(), NATURALSET::empty());
		// }
		// DELETE(non_block_predicates);
		// //delete incorrect previous-references in the new net
		// new_net.delete_previous_node( already_correct_positioned_blocks);
		// create_next_references(new_net);
		// create_preconditions_and_update_postconditions(new_net,
		// object,global_precondition);
		// insert_the_new_net(net, new_net, node_nr);
		// new_net.subdeepdelete();
		//
		// return TRUE;

	}


	//
	// @Override
	// public Object getObjectInstance(Object boostedNode, Name ignoredName,
	// Context ignoredContext, Hashtable<?, ?> environmentWithPlan)
	// throws Exception {
	// return expand(boostedNode, environmentWithPlan);
	// // Plan<Table> plan = Planning
	// // .getPlanFromHashtable(environmentWithPlan);
	// // BoostedNodeGraph result = transformWithException(
	// // (BoostedNode) boostedNode, plan);
	// //
	// // return result;
	// }

	private BoostedNodeGraph expand(Object boostedNode, Object hashtableWithPlan) throws Exception {
		Hashtable<?, ?> environmentWithPlan = (Hashtable) hashtableWithPlan;

		Plan<Table> plan = Planning
				.getPlanFromHashtable(environmentWithPlan);
		BoostedNodeGraph result = transformWithException(
				(BoostedNode) boostedNode, plan);

		return result;

	}
	/**
	 * Expands the node according to this type
	 * 
	 * @param planNode
	 *            the inspected node
	 * @param plan
	 *            the plan of the node
	 * @return the solution/expansion of the node
	 * @throws Exception
	 */
	private BoostedNodeGraph transformWithException(BoostedNode planNode,
			Plan<Table> plan)
			throws Exception {
		Operation<Table> node = Operation
				.getOperation((BoostedNode) planNode);
		Table postcondition = node.getPostcondition();
		
		if (Blockworld.OPERATOR_TO_SOLVE.equals(type)) {
			// throw new IllegalStateException("not yet implemented " + type);
			if (postcondition == null) {
				getLog().info("null postcondition, nothing needs to be done");
				return null;
			} else if (node.isNoNeedToSolve() || node.isSolved()) {
				return null;
			} else if (postcondition.hasBlockPredicate()) {
				return solveWithBlocks(node);
			} else if (postcondition.isElementary()) {
				return solveElementary(node);
			} else {
				getLog().info("could not expand TO_SOLVE node");
				// no solving algorithm yet
				return null;
			}
		} else if (Blockworld.OPERATOR_MOVE_BLOCK.equals(type)) {
			return moveBlock(node, plan);//
		} else if (Blockworld.OPERATOR_MAKE_SURFACE.equals(type)) {
			return makeSurface(node);
		} else if (Blockworld.OPERATOR_GRAB_BLOCK.equals(type)) {
			return grabBlock(node);
		} else if (Blockworld.OPERATOR_MOVE_HAND.equals(type)) {
			return moveHand(node);
		} else if (Blockworld.OPERATOR_FREE_THE_BLOCK.equals(type)) {
			return freeTheBlock(node);
		} else if (Blockworld.OPERATOR_REMOVE_BLOCK.equals(type)) {
			return removeBlock(node, plan);
		} else {
			throw new IllegalStateException("not yet implemented: " + type);
		}
	}

	/**
	 * Algorithm: create a NEW HAND, GRAB the block with the new hand, MOVE the hand to the new position, TURN the hand into the appropriate orientation, MOVE
	 * DOWN the hand, RELEASE the block, forget the new hand.
	 * 
	 * 
	 * @param plan
	 *            the plan the node belongs to
	 * @return the solution/expansion
	 */
	public BoostedNodeGraph moveBlock(Operation<Table> node, Plan<Table> plan)
			throws Exception {
		// getLog().warn("conditions are not yet updated in moveBlock");
		// see special_solve_MOVE_BLOCK

		// expand the node only when there is a global precondition.
		// This ensures that first nodes, at the beginning of the plan are
		// expanded

		// TODO activate the following lines
		// if (node.getGlobalPrecondition() == null) {
		// return null;
		//
		// }

		int nextPriority = node.getSolvingPriority() - 1;
		Block blockToMove = node.getOperatorDetails();

		Hand hand = Hand.newEmptyHand(X_OF_NEW_HAND, Y_OF_NEW_HAND);

		BoostedNodeGraph result = Operation.createBoostedNodeGraph(
				// do i need the node.getPrecondition() as precondition???
				BlockworldOperation.newVirtualHand(nextPriority, hand),
				BlockworldOperation.grabBlock(nextPriority, hand,
						blockToMove),
				BlockworldOperation.moveHand(nextPriority, hand,
						blockToMove.getPosition(), Boolean.TRUE),
				BlockworldOperation.turn(nextPriority, hand,blockToMove.getOrientation()),
				BlockworldOperation.moveDown(nextPriority, hand),
				BlockworldOperation.release(nextPriority, hand),
				BlockworldOperation.deleteVirtualHand(nextPriority, hand));

		return result;
	}


	/**
	 * Creates a graph that provides a given layout
	 * 
	 * @return the solution/expansion
	 */
	public BoostedNodeGraph makeSurface(Operation<Table> node)
			throws Exception {

		Table globalPrecondition = node.getGlobalPrecondition();
		Table postcondition = node.getPostcondition();

		if (globalPrecondition == null || postcondition == null) {
			return null; // nothing can be done (yet)
		} else if (globalPrecondition.implies(postcondition).isEmpty()) {
			return null; // nothing needs to be done
		} else {
			/*
			 * manipulate the table, so that the postcondition of the object is
			 * realized ( the postcondition must not be to difficult)
			 */
			Table tableWithFreePlaces = node.getOperatorDetails();

			Collection<FreePlace> freePlaces = tableWithFreePlaces
					.getFreePlaces();

			// TODO ("improve or remove the forbidden places");
			Table forbiddenPlacesCondition = tableWithFreePlaces
					.createEmptyCondition();
			forbiddenPlacesCondition
					.addTableElements((Collection) tableWithFreePlaces
					.createForbiddenPlaces(freePlaces));

			// all blocks that have to be remove
			HashSet<Block> blocksToRemove = new HashSet<Block>();
			// all blocks that lie directly on places that should be free
			blocksToRemove.addAll(globalPrecondition
					.getBottommostBlocks(freePlaces));
			// all blocks that lie directly over blocks that should be free
			// possible improvement: update forbidden_places_condition
			blocksToRemove
					.addAll(globalPrecondition
							.getBlocksDirectlyOverBlocks(postcondition
									.getFreeBlocks()));

			// REMOVE all these blocks to other places (parallel)
			int solvingPriority = node.getSolvingPriority() - 1;
			BoostedNodeGraph result = new BoostedNodeGraph();

			// Add a toSolve to not forget the forbidden places
			// TODO maybe the forbidden places should be placed elsewhere
			Operation<Table> forbiddenReminder = BlockworldOperation
					.toSolve(solvingPriority, forbiddenPlacesCondition);
			// maybe additional in precondition???

			// add remove the blocks before that node
			for (Block blockToRemove : blocksToRemove) {
				result.addNode(Operation.createBoostedNode(BlockworldOperation
						.removeBlock(solvingPriority, blockToRemove)));

			}
			return result;

		}
	}

	/**
	 * Creates a graph algorithm that grabs the block
	 * 
	 * @return the solution/expansion
	 * @param operation
	 *            the operation to solve
	 */
	public BoostedNodeGraph grabBlock(Operation<Table> operation)
			throws Exception {

		int solvingPriority = operation.getSolvingPriority();
		int nextPriority = solvingPriority - 1;

		Hand handWithBlockToGrab = operation.getOperatorDetails();
		Table globalPrecondition = operation.getGlobalPrecondition();
		// GRAB the block only, when enough informations about the block are
		// known
		if (globalPrecondition == null) {
			return null;
			// because not enough is known about the other blocks
		}

		Block blockToGrab = globalPrecondition.findSubcondition(new Block(
				handWithBlockToGrab.getNameOfBlock()));
		if (blockToGrab == null){
			getLog().info("block to grab ", handWithBlockToGrab.getNameOfBlock(), " not available in global precondition ", globalPrecondition);
			return null;
		}
		
		BoostedNodeGraph result = new BoostedNodeGraph();
		Hand blockHoldingHand = globalPrecondition.getHandHoldingBlock(blockToGrab);
		
		
		if (blockHoldingHand != null){
			if (blockHoldingHand.getName().equals(handWithBlockToGrab.getName())){
				// since the hand already holds the block, nothing is to do,
				// just make a toSolve node with the hand holding the block to
				// keep the condition
				Operation<Table> toSolve = (Operation<Table>) operation
						.clone();
				toSolve.setOperator(Blockworld.OPERATOR_TO_SOLVE);
				toSolve.setOperatorDetails(null);
				toSolve.setSolvingPriority(nextPriority - 1);//less important
				Set<TableElement> tableElements = toSolve.getPostcondition()
						.getTableElements();
				tableElements.clear();
				tableElements.add((Hand) handWithBlockToGrab.cloneMe());

				result.addNode(Operation.createBoostedNode(toSolve));
			} else {
				// Release the block with the one hand and try to grab it again
				// other
				Operation<Table> clonedOperation = (Operation<Table>) operation
						.clone();
				clonedOperation.setSolvingPriority(nextPriority);
				
				result = Operation
.createBoostedNodeGraph(BlockworldOperation
						.releaseBlock(nextPriority, blockToGrab));
				
			}
		} else {
			// free the block, move the hand to the block, down and grab

			result = Operation
					.createBoostedNodeGraph(

					BlockworldOperation.freeBlock(nextPriority, blockToGrab),
					BlockworldOperation.moveHand(nextPriority,
									handWithBlockToGrab,
									blockToGrab.getPosition(), Boolean.FALSE),
							BlockworldOperation.grab(nextPriority,
									handWithBlockToGrab));

		}
		// result.simpleDisplay();
		return result;
		
		// else if ( (block->type == anyblock)
		// ||
		// ( block->is_down()
		// &&
		// ( (block->x == 0)
		// ||
		// (block->y == 0)
		// ||
		// (block->orientation== any)
		// )
		// )
		// )
		// result = FALSE;
		
		
	}

	// BOOLEAN special_solve_GRAB_BLOCK( BLOCKWORLD_PLAN& net,
	// BLOCKWORLD_PLAN_OBJECT& object, NATURAL node_nr)
	// {
	// TABLE_CONDITION *whole_precondition = (TABLE_CONDITION
	// *)(POINTER)object.actual_precondition();
	// TABLE_CONDITION *new_postcondition;
	// TABLE_CONDITION *new_precondition;
	// BLOCKWORLD_PLAN_OBJECT *new_node_element;
	// BOOLEAN result= TRUE;;
	//
	// NATURAL blocknr = ( (OPERATOR *)object.element)->n2;
	// NATURAL handnr = ( (OPERATOR *)object.element)->n1;
	// HAND *hand = object.find_hand_in_preconditions( handnr);
	// BLOCK *block = object.find_block_in_preconditions( blocknr);
	// HAND * other_hand = object.find_hand_with_block_in_global_precondition(
	// blocknr);
	//
	// else
	// {
	// /* GRAB the block with the hand
	// ============================
	// RELEASE any block that is held by the hand # FREE the block
	// MOVE the hand to the block #
	// GRAB the block
	// */
	//
	// result = TRUE;
	// NATURAL hand_new_x = block->x + block->grip_x();
	// NATURAL hand_new_y = block->y + block->grip_y();
	//
	// NATURALSET *old_previous =net.previous(node_nr);
	//
	// new_postcondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_postcondition->add_hand(handnr,0,0,TRUEFALSE,0,none);
	// new_node_element = BLOCKWORLD_PLAN_OBJECT::create(
	// object.deepcopy_global_precondition(), object.deepcopy_precondition(),
	// RELEASE_BLOCK, handnr,new_postcondition,NULL);
	// new_node_element->solving_priority = 0;
	// net.insert_before(node_nr, new_node_element);
	//
	// new_precondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_precondition->add_hand( handnr,0,0,TRUEFALSE,0,none);
	// new_postcondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_postcondition->add_hand(handnr,hand_new_x,hand_new_y,FALSE,0,none);
	// new_node_element = BLOCKWORLD_PLAN_OBJECT::create( NULL,
	// new_precondition,MOVE_HAND, handnr, hand_new_x, hand_new_y,any,FALSE,
	// new_postcondition,NULL);
	// new_node_element->solving_priority = 0;
	// net.insert_before(node_nr, new_node_element);
	//
	// new_postcondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_postcondition->add_FREE_BLOCK(blocknr);
	// new_node_element = BLOCKWORLD_PLAN_OBJECT::create(
	// object.deepcopy_global_precondition(),
	// NULL,FREE_THE_BLOCK,blocknr,new_postcondition,NULL );
	// new_node_element->solving_priority = object.small_priority_decrement();
	// net.insert_between(*old_previous, NATURALSET::only(node_nr),
	// new_node_element);
	//
	// new_precondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_precondition->add_hand( handnr,hand_new_x,hand_new_y,FALSE,0,none);
	// new_precondition->add_block( blocknr,block->x, block->y,block->type,
	// block->orientation,FALSE,NATURALSET::only(0));
	// new_precondition->add_FREE_BLOCK(blocknr);
	// new_precondition->add_FORBIDDEN_PLACE( block->type, block->orientation,
	// block->x, block->y);
	// new_postcondition = (( TABLE_CONDITION
	// *)net.CONDITION_dummy)->deepcopy_();
	// new_postcondition->add_hand_has_block(handnr,blocknr);
	// DELETE(old_previous);
	// object.change(NULL,new_precondition,GRAB, handnr,new_postcondition,
	// NULL);
	// object.solving_priority = 0;
	// }
	// return result;
	// }

	/**
	 * Creates a graph algorithm that moves a hand
	 * 
	 * @return the solution/expansion
	 * @param operation
	 *            the operation to solve
	 */
	public BoostedNodeGraph moveHand(Operation<Table> operation)
			throws Exception {

		int solvingPriority = operation.getSolvingPriority();
		int nextPriority = solvingPriority - 1;

		Hand handToMove = operation.getOperatorDetails();

		ArrayList<Operation<?>> operations = new ArrayList<Operation<?>>();
		operations.add(BlockworldOperation.moveUp(nextPriority, handToMove));
		operations.add(BlockworldOperation.move(nextPriority, handToMove,
				handToMove.getPosition()));
		if (handToMove.getGrip() != null) {
			operations.add(BlockworldOperation.turn(nextPriority, handToMove,
					handToMove.getGrip()));
		}

		if (Boolean.FALSE.equals(handToMove.getUp())) {
			operations.add(BlockworldOperation.moveDown(nextPriority,
					handToMove));
		}

		BoostedNodeGraph result = Operation.createBoostedNodeGraph(operations);

		return result;

	}

	/**
	 * Creates a graph algorithm that grabs the block
	 * 
	 * @param operation
	 *            the operation to solve
	 * @return the solution/expansion
	 */
	public BoostedNodeGraph freeTheBlock(Operation<Table> operation)
			throws Exception {

		int solvingPriority = operation.getSolvingPriority();
		int nextPriority = solvingPriority - 1;

		Block blockToSetFree = operation.getOperatorDetails();
		Table globalPrecondition = operation.getGlobalPrecondition();

		if (globalPrecondition == null) {
			return null;
			// nothing can be done yet
		} else {
			Set<Block> blocks = globalPrecondition
					.getBlocksDirectlyOverBlocks(Arrays.asList(blockToSetFree));
			if (blocks.isEmpty()) {
				return null;
				// nothing needs to be done, the block is free already
			} else {
				// REMOVE parallel all blocks that are over the block

				BoostedNodeGraph result = new BoostedNodeGraph();
				Operation<Table> freeBlock = BlockworldOperation
						.freeBlock(nextPriority - 10, blockToSetFree);
				// to not forget that the block should be free
				BoostedNode freeBlockNode = result.addNodeWithValue(freeBlock);

				for (Block block : blocks) {
					Operation<Table> removeBlock = BlockworldOperation
							.removeBlock(nextPriority, block);
					BoostedNode removeBlockNode = result
							.addNodeWithValue(removeBlock);
					result.addEdge(removeBlockNode, freeBlockNode);
				}

				return result;
			}
		}
	}

	/**
	 * Creates a graph algorithm that removes the block
	 * 
	 * @param operation
	 *            the operation to solve
	 * @param plan
	 *            the plan the operation belongs to
	 * @return the solution/expansion
	 * 
	 */
	public BoostedNodeGraph removeBlock(Operation<Table> operation, Plan<Table> plan)
			throws Exception {

		int solvingPriority = operation.getSolvingPriority();
		int nextPriority = solvingPriority - 1;

		Block blockToRemove = operation.getOperatorDetails();
		Table globalPrecondition = operation.getGlobalPrecondition();

		if (globalPrecondition == null) {
			return null;
			// nothing can be done yet
		} else {
		
			Hand hand = Hand.newEmptyHand(X_OF_NEW_HAND, Y_OF_NEW_HAND);

			// get all occupied places of the plan
			HashSet<Place> occupiedPlaces = new HashSet<Place>();
			for (BoostedNode node : plan.getGraph().getAllVertices()) {
			    Operation operat = Operation.getOperation(node);
				BlockworldOperation /* Operation<Table, String> */op = (BlockworldOperation) operat;
				op.addOccupiedPlacesTo(occupiedPlaces);
				
			}
			// find a place that is not occupied
			Integer x = null;
			Integer y = null;
			for (int yy = 0; yy < globalPrecondition.getMax_y(); yy++) {
				for (int xx = 0; xx < globalPrecondition.getMax_x(); xx++) {
					if (! occupiedPlaces.contains(new Place(xx, yy))){
						x = xx;
						y = yy;
						break;
					}
				}
				if (x != null){
					break;
				}
			}
			
			if (x == null) {
				// use one default place.
				// todo better use an onoccupied place of the current
				// precondition
				y = globalPrecondition.getMax_y();
				x = globalPrecondition.getMax_x();
			}

			Block targetBlock = blockToRemove.cloneMe();
			targetBlock.setX(x);
			targetBlock.setY(y);
			targetBlock.setOrientation(Orientation.Z);
			targetBlock.setUp(Boolean.FALSE);
			targetBlock.setOverBlocks(new HashSet<Object>());

			// TODO better: only grab the block and move up. Later can be
			// decided where the block is to be dropped

			BoostedNodeGraph result = Operation.createBoostedNodeGraph(

					BlockworldOperation.moveBlock(nextPriority, targetBlock)
			);

			return result;
			
			
		}

	}

	@Override
	protected BoostedNodeGraph callImpl(Object... parameters) throws Exception {

		return expand(parameters[0], parameters[1]);
	}

}
