package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.awt.Dimension;
import java.util.Collection;

import net.sf.gluebooster.java.booster.basic.math.Operation;
import net.sf.gluebooster.java.booster.basic.math.Operator;
import net.sf.gluebooster.java.booster.essentials.meta.HasName;
import net.sf.gluebooster.java.booster.essentials.utils.Check;

/**
 * The operations of the blockworld.
 * 
 * @author CBauer
 * 
 * @defaultParamText nextPriority The priority of the operation.
 * @defaultParamText solvingPriority The priority of the operation.
 * @defaultParamText precondition The precondition of the operation.
 * @defaultParamText postcondition The postcondition of the operation.
 * @defaultParamText operatorDetails details of the operator of the operation.
 * @defaultParamText worldInstanceDescription the actual world (table)
 * @defaultParamText operator the operator of the operation
 * 
 * 
 * 
 * 
 */
public class BlockworldOperation extends Operation<Table> {

	/**
	 * Singleton instance.
	 */
	private static BlockworldOperation singleton = new BlockworldOperation();

	public BlockworldOperation() {
		super();
	}

	/**
	 * 
	 */
	public BlockworldOperation(Operator operator,
			HasName operatorDetails, int solvingPriority,
			Table precondition, Table postcondition) {
		super(operator, operatorDetails, solvingPriority, precondition,
				postcondition);
	}

	public BlockworldOperation(Operator operator,
			HasName operatorDetails, int solvingPriority,
			Table postcondition) {
		super(operator, operatorDetails, solvingPriority, postcondition);
	}

	public BlockworldOperation(Operator operator,
			HasName operatorDetails, int solvingPriority) {
		super(operator, operatorDetails, solvingPriority);
	}

	/**
	 * Create the operation that a node is still to solve.
	 * 
	 * @param riddle
	 *            the target configuration of the table
	 * @return the created operation
	 */
	public static BlockworldOperation toSolve(int nextPriority,
			Table riddle) {
		return new BlockworldOperation(Blockworld.OPERATOR_TO_SOLVE, null,
				nextPriority, riddle);
	}

	/**
	 * Create the operation that a block is to be moved to a given position.
	 * 
	 * @param targetPosition
	 *            the wanted position
	 * @return the created operation
	 */
	public static BlockworldOperation moveBlock(int nextPriority,
			Block targetPosition) throws Exception {

		Check.notNull(targetPosition.getX(), "X", targetPosition);
		Check.notNull(targetPosition.getY(), "Y", targetPosition);
		Check.notNull(targetPosition.getOrientation(), "Orientation",
				targetPosition);

		return new BlockworldOperation(Blockworld.OPERATOR_MOVE_BLOCK,
				targetPosition.cloneMeOnlyName().setPosition(
						targetPosition.getX(), targetPosition.getY(),
						targetPosition.getOrientation()),
				nextPriority);
	}

	/**
	 * Create the operation that a block is to be removed from its position.
	 * 
	 * @param block
	 *            the block to be removed from its place
	 * @return the created operation
	 * 
	 */
	public static BlockworldOperation removeBlock(int nextPriority, Block block)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_REMOVE_BLOCK,
				block.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that a block is to be released from a holding hand.
	 * 
	 * @param block
	 *            the block to be released
	 * @return the created operation
	 */
	public static BlockworldOperation releaseBlock(int nextPriority, Block block)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_RELEASE_BLOCK,
				block.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that a block is to be freed from blocks above him.
	 * 
	 * @param block
	 *            the block to be freed
	 * @return the created operation
	 */
	public static BlockworldOperation freeBlock(int nextPriority, Block block)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_FREE_THE_BLOCK,
				block.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that given table surface structure is to be created.
	 * 
	 * @param target
	 *            the wanted layout of the table
	 * @return the created operation
	 */
	public static BlockworldOperation makeSurface(int nextPriority,
			Table target) {
		return new BlockworldOperation(Blockworld.OPERATOR_MAKE_SURFACE,
				target, nextPriority, target);
	}

	/**
	 * Create the operation that a new virtual hand is to be created.
	 * 
	 * @param newHand
	 *            the hand to be created
	 * @return the created operation
	 */
	public static BlockworldOperation newVirtualHand(int nextPriority,
			Hand newHand) throws Exception {
		return new BlockworldOperation(Blockworld.OPERATOR_NEW_VIRTUAL_OBJECT,
				(Hand) newHand.cloneMe(), nextPriority);
	}

	/**
	 * Create the operation that a virtual hand is no longer needed.
	 * 
	 * @param hand
	 *            the hand to be deleted
	 * @return the created operation
	 */
	public static BlockworldOperation deleteVirtualHand(int nextPriority,
			Hand hand) throws Exception {
		return new BlockworldOperation(
				Blockworld.OPERATOR_DELETE_VIRTUAL_OBJECT,
				(Hand) hand.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that a hand should grab a block.
	 * 
	 * @param hand
	 *            the hand that should grab the block
	 * @param block
	 *            the block to be grabbed
	 * @return the created operation
	 */
	public static BlockworldOperation grabBlock(int nextPriority,
			Hand hand, Block block) throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_GRAB_BLOCK, ((Hand) hand.cloneMeOnlyName()).setHasBlock(block.getName()), nextPriority);
	}

	/**
	 * Move an up Hand
	 * 
	 * @param hand
	 *            the hand to move
	 * @param newPosition
	 *            the target position of the hand
	 * @return the created operation
	 */
	public static BlockworldOperation move(int nextPriority, Hand hand,
			Dimension newPosition) throws Exception {

		Check.notNull(newPosition, "newPosition");

		return new BlockworldOperation(Blockworld.OPERATOR_MOVE, ((Hand) hand.cloneMeOnlyName()).setPosition(newPosition), nextPriority);
	}

	/**
	 * Create the operation that a hand should grab (the block at the position of the hand).
	 * 
	 * @param hand
	 *            the hand to grab
	 * @return the created operation
	 */
	public static BlockworldOperation grab(int nextPriority, Hand hand)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_GRAB,
				(Hand) hand.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that hand should be moved to a new position.
	 * 
	 * @param hand
	 *            the hand to move
	 * @param newPosition
	 *            the target position of the hand
	 * @param up
	 *            should the hand be up (true) or down (false)
	 * @return the created operation
	 */
	public static BlockworldOperation moveHand(int nextPriority, Hand hand,
			Dimension newPosition, Boolean up) throws Exception {

		Check.notNull(newPosition, "newPosition");

		BlockworldOperation result = new BlockworldOperation(
				Blockworld.OPERATOR_MOVE_HAND, ((Hand) hand.cloneMeOnlyName())
						.setPosition(newPosition), nextPriority);

		Hand newHand = result.getOperatorDetails();
		newHand.setUp(up);
		return result;

	}

	/**
	 * Create the operation that a hand has to turn (the held block) to a given orientation.
	 * 
	 * @param hand
	 *            the hand that should turn
	 * @param orientation
	 *            the new orientation of the hand
	 * @return the created operation
	 * 
	 */
	public static BlockworldOperation turn(int nextPriority, Hand hand,
			Orientation orientation) throws Exception {
		Check.notNull(orientation, "orientation");

		return new BlockworldOperation(Blockworld.OPERATOR_TURN, ((Hand) hand
				.cloneMeOnlyName()).setGripReturnThis(orientation), nextPriority);
	}

	/**
	 * Create the operation that a hand is to move down.
	 * 
	 * @param hand
	 *            the hand to move
	 * @return the created operation
	 */
	public static BlockworldOperation moveDown(int nextPriority, Hand hand)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_MOVE_DOWN,
				(Hand) hand.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that hand is to move up.
	 * 
	 * @param hand
	 *            the hand to move
	 * @return the created operation
	 */
	public static BlockworldOperation moveUp(int nextPriority, Hand hand)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_MOVE_UP,
				(Hand) hand.cloneMeOnlyName(), nextPriority);
	}

	/**
	 * Create the operation that a hand has to release the held block.
	 * 
	 * @param hand
	 *            the hand that should release its block
	 * @return the created operation
	 */
	public static BlockworldOperation release(int nextPriority, Hand hand)
			throws Exception {

		return new BlockworldOperation(Blockworld.OPERATOR_RELEASE,
				(Hand) hand.cloneMeOnlyName(), nextPriority);
	}

	@Override
	public void checkOperator() throws Exception {
		Operator operator = getOperator();
		if (Blockworld.OPERATOR_TO_SOLVE.equals(operator)) {
			checkToSolveOperator();
		} else {
			// getLog().info("check operator not implemented for " + operator);
		}

	}

	/**
	 * Checks that the postconditions of this to-solve-operator are good enough described.
	 * 
	 * @throws Exception
	 *             if there are not enough post conditions.
	 */
	private void checkToSolveOperator() throws Exception {
		Check.notNull(getPostcondition(), "postcondition");
		Table table = getPostcondition();
		Check.notEmpty(table.getTableElements(), "tableElements");
	}


	/**
	 * Gets a default delete condition
	 * 
	 * @param operation
	 *            should be delete virtual object
	 * @param worldInstanceDescription
	 *            the world
	 * @return null if the operation is not delete virtual object
	 * @throws Exception
	 */
	public static Table getDefaultDeletecondition(
			Operation<Table> operation, Table worldInstanceDescription)
			throws Exception {
		worldInstanceDescription.getTableElements().clear();

		if (Blockworld.OPERATOR_DELETE_VIRTUAL_OBJECT
				.equals(operation.getOperator())) {
			TableElement handToDelete = operation.getOperatorDetails();
			Table table = worldInstanceDescription
					.addTableElement(handToDelete);
			return table;
		} else {
			// nothing to do
			return null;
		}
	}

	/**
	 * Creates a default postcondition
	 * 
	 * @param node
	 *            the inspected node
	 * @return null if no default postcondition exists
	 */
	public static Table getDefaultPostcondition(Operation<Table> node,
			Table worldInstanceDescription) throws Exception {
		Operator operator = node.getOperator();
		Object operatorDetails = node.getOperatorDetails();

		Hand hand = null;

		Table resultTable = worldInstanceDescription;
		resultTable.getTableElements().clear();

		// may hold a block
		Block blockOfHand = null;
		if (operatorDetails != null && operatorDetails instanceof Hand) {
			hand = node.getOperatorDetails(); // may hold block
			hand = hand.cloneMe();// to prevent side effects of changes
			resultTable.addTableElement(hand);
			if (hand.getNameOfBlock() != null) {
				blockOfHand = new Block(hand.getNameOfBlock(), hand.getX(),
						hand.getY(), null, hand.getGrip(), hand.getUp());
				resultTable.addTableElement(blockOfHand);
			}
		}

		Block block = null;
		if (operatorDetails != null && operatorDetails instanceof Block) {
			block = node.getOperatorDetails();
			block = block.cloneMe();
			resultTable.addTableElement(block);
		}

		if (Blockworld.OPERATOR_DELETE_VIRTUAL_OBJECT.equals(operator)) {
			// no default postcondition
			return null;
		} else if (Blockworld.OPERATOR_FREE_THE_BLOCK.equals(operator)) {
			// only create a free block condition
			resultTable.getTableElements().clear();
			resultTable.addTableElement(new FreeBlock(block));
		} else if (Blockworld.OPERATOR_GRAB.equals(operator)) {
		} else if (Blockworld.OPERATOR_GRAB_BLOCK.equals(operator)) {
		} else if (Blockworld.OPERATOR_MAKE_SURFACE.equals(operator)) {
			// no default postcondition
			return null;
		} else if (Blockworld.OPERATOR_MOVE.equals(operator)) {
		} else if (Blockworld.OPERATOR_MOVE_BLOCK.equals(operator)) {
		} else if (Blockworld.OPERATOR_MOVE_DOWN.equals(operator)) {
		} else if (Blockworld.OPERATOR_MOVE_HAND.equals(operator)) {
		} else if (Blockworld.OPERATOR_MOVE_UP.equals(operator)) {
		} else if (Blockworld.OPERATOR_NEW_VIRTUAL_OBJECT.equals(operator)) {
			// because until now only hands are virtual objects
		} else if (Blockworld.OPERATOR_RELEASE.equals(operator)) {
		} else if (Blockworld.OPERATOR_RELEASE_BLOCK.equals(operator)) {
			hand.setNameOfBlock(null);
			hand.setHasBlock(false);
		} else if (Blockworld.OPERATOR_REMOVE_BLOCK.equals(operator)) {
			// no default postcondition
			return null;
		} else if (Blockworld.OPERATOR_TO_SOLVE.equals(operator)) {
			return null;
			// no default postcondition
		} else if (Blockworld.OPERATOR_TURN.equals(operator)) {
			// hand has block
			// new_postcondition->add_block( block->nr, new_x, new_y,
			// block->type, orientation, TRUE, empty_naturalset);
		} else {
			singleton.getLog().warn(
					"getDefaultPostcondition not yet implemented for "
							+ node.getOperator());
			return null;
		}

		return resultTable;
	}

	/**
	 * Creates a default precondition
	 * 
	 * @param operation
	 *            the inspected operation
	 * @return null if no default precondition exists
	 */
	public static Table getDefaultPrecondition(
			Operation<Table> operation,
			Table worldInstanceDescription) throws Exception {

		Table result = null;
		// create a template of the table (without elements)
		Table table = worldInstanceDescription;
		table.getTableElements().clear();

		Operator operator = operation.getOperator();
		if (Blockworld.OPERATOR_DELETE_VIRTUAL_OBJECT.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_FREE_THE_BLOCK.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_GRAB.equals(operator)) {

			// the hand must be down and hold no block
			Hand hand = operation.getOperatorDetails();
			Hand downHand = new Hand(hand.getName());
			downHand.setUp(false);
			downHand.setHasBlock(false);
			// TODO there must be a free block at the position

			table.addTableElement(downHand);
			result = table;

		} else if (Blockworld.OPERATOR_GRAB_BLOCK.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_MAKE_SURFACE.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_MOVE.equals(operator)
				|| Blockworld.OPERATOR_TURN.equals(operator)
				|| Blockworld.OPERATOR_MOVE_DOWN.equals(operator)) {

			// the hand must be up
			Hand hand = operation.getOperatorDetails();
			Hand upHand = new Hand(hand.getName());
			upHand.setUp(true);
			table.addTableElement(upHand);

			result = table;
		} else if (Blockworld.OPERATOR_MOVE_UP.equals(operator)) {

			// the hand must be down
			Hand hand = operation.getOperatorDetails();
			Hand downHand = new Hand(hand.getName());
			downHand.setUp(false);
			table.addTableElement(downHand);

			result = table;

		} else if (Blockworld.OPERATOR_MOVE_BLOCK.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_MOVE_HAND.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_NEW_VIRTUAL_OBJECT.equals(operator)) {
			// no precondition necessary
		} else if (Blockworld.OPERATOR_RELEASE.equals(operator)) {

			// the hand must be down and hold a block
			Hand hand = operation.getOperatorDetails();
			Hand downHand = new Hand(hand.getName());
			downHand.setUp(false);
			downHand.setHasBlock(true);
			table.addTableElement(downHand);
			result = table;
		} else if (Blockworld.OPERATOR_REMOVE_BLOCK.equals(operator)) {
			// no precondition necessary

		} else if (Blockworld.OPERATOR_TO_SOLVE.equals(operator)) {
			// no precondition necessary
		} else {
			singleton.getLog().warn(
					"getDefaultPrecondition is not yet adapted. " + operator);
		}
		return result;
	}

	/**
	 * Adds all occupied places of the global precondition to a collection
	 * 
	 * @param occupiedPlaces
	 *            the places are added here
	 */
	public void addOccupiedPlacesTo(Collection<Place> occupiedPlaces) {
		Table table = getGlobalPrecondition();
		if (table != null) {
			occupiedPlaces.addAll(table.getOccupiedPlaces());
		}
		table = getGlobalPostcondition();
		if (table != null) {
			occupiedPlaces.addAll(table.getOccupiedPlaces());
		}
		table = getPostcondition();
		if (table != null) {
			occupiedPlaces.addAll(table.getOccupiedPlaces());
		}

	}
}
