package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.rmi.server.UID;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import net.sf.gluebooster.java.booster.essentials.math.Condition;
import net.sf.gluebooster.java.booster.essentials.utils.Check;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;


/**
 * A hand grabs the blocks and moves them.
 * 
 * TODO refactor boostedobject, so that the name is generic.
 * 
 * @author CBauer
 * @defaultParamText x the x-coordinate
 * @defaultParamText y the y-coordinate
 * 
 */
public class Hand extends AbstractTableElement<Hand> {

	/**
	 * Is the hand up or down.
	 */
	private Boolean up;

	/**
	 * The orientation of the grip (of the block)
	 */
	private Orientation grip;

	/**
	 * The name of the block this hand holds.
	 */
	private Object nameOfBlock;

	/**
	 * Does this hand hold a block.
	 */
	private Boolean hasBlock;

	public Hand() {
		setCompareById(true);
		// the name is the id
	}

	/**
	 * Constructor with name
	 * 
	 * @param name
	 *            the name of this hand
	 */
	public Hand(Object name) {
		this();
		setName(name);
	}

	/**
	 * Constructor with name and position
	 * 
	 * @param name
	 *            the name of this hand
	 * @param x
	 *            the x-coordinate
	 * @param y
	 *            the y-coordinate
	 */
	public Hand(Object name, int x, int y) {
		this();
		setName(name);
		setUp(true);
		setX(x);
		setY(y);
		setHasBlock(false);
	}

	/**
	 * Create an empty hand without block
	 * 
	 * @return the new hand
	 */
	public static Hand newEmptyHand(int x, int y) {
		Hand result = new Hand(new UID().toString());
		result.setHasBlock(false);
		result.setUp(true);
		result.setX(x);
		result.setY(y);

		return result;
	}

	public Boolean getUp() {
		return up;
	}

	public void setUp(Boolean up) {
		this.up = up;
	}

	public Orientation getGrip() {
		return grip;
	}

	public Hand setGripReturnThis(Orientation grip) {
		setGrip(grip);
		return this;
	}

	public void setGrip(Orientation grip) {
		this.grip = grip;
	}

	public Object getNameOfBlock() {
		return nameOfBlock;
	}

	public void setNameOfBlock(Object nameOfBlock) {
		this.nameOfBlock = nameOfBlock;
	}

	@Override
	public String toString() {
		StringBuilder result = new StringBuilder("hand " + getName());
		if (getX() != null || getY() != null) {
			result.append(" (");
			result.append((getX() != null) ? getX() : "?");
			result.append(", ");
			result.append((getY() != null) ? getY() : "?");
			result.append(")");
		}

		if (Boolean.TRUE.equals(up)) {
			result.append(" up");
		} else if (Boolean.FALSE.equals(up)) {
			result.append(" down");
		}

		if (hasBlock == null) {
			// result.append("may hold a block ");
		} else if (Boolean.FALSE.equals(hasBlock)) {
			result.append(" empty");
		} else { // hasBloc == true
			if (nameOfBlock == null)
				result.append(" some block");
			else {
				result.append(" ").append(nameOfBlock);
				
			}
		}

		if (grip != null)
			result.append(" ").append(grip);

		return result.toString();
	}

	public Boolean getHasBlock() {
		return hasBlock;
	}

	public void setHasBlock(Boolean hasBlock) {
		this.hasBlock = hasBlock;
	}

	public Hand setHasBlock(Object newNameOfBlock) throws Exception {
		nameOfBlock = newNameOfBlock;
		setHasBlock(Boolean.TRUE);
		return this;
	}


	/**
	 * Sets the position of this hand
	 * 
	 * @param up
	 *            is the hand up
	 * @return this
	 */
	public Hand setPosition(Integer x, Integer y, Boolean up) throws Exception {
		setPosition(x, y);
		setUp(up);
		return this;
	}

	/**
	 * TODO refactor into boostedObject, so that the names work as ids.
	 */
	@Override
	public void setName(Object name) {
		super.setName(name);
		super.setId(name);
	}

	@Override
	public Collection isInConflict(Condition condition) throws Exception {
		boolean result = false;
		if (condition instanceof Hand) {
			Hand hand = (Hand) condition;
			if (getId().equals(hand.getId())) {
				result = !couldBe(hand) || !hand.couldBe(this);
			}
		}

		if (result) {
			return Arrays.asList("some conflict");
		} else {
			return Collections.EMPTY_LIST;
		}

	}

	@Override
	protected boolean couldBe(Hand hand) {
		if (!super.couldBe(hand))
			return false;

		if (!couldBe(getUp(), hand.getUp(), null))
			return false;

		if (!couldBe(hasBlock, hand.hasBlock, null))
			return false;

		if (Boolean.TRUE.equals(hasBlock)
				&& !couldBe(nameOfBlock, hand.nameOfBlock, null))
			return false;

		if (!couldBe(grip, hand.grip, Orientation.ANY))
			return false;

		return true;

	}

	@Override
	public void checkThatAllStatesAreKnown() throws IllegalStateException {
		// Refactor.notNull(grip, "grip");
		// the grip may be null, the block should be turned
		Check.notNull(hasBlock, "hasBlock");
		Check.notNull(up, "up");
		if (Boolean.TRUE.equals(hasBlock)) {
			Check.notNull(nameOfBlock, "nameOfBlock");
		}
	}

	@Override
	public String asSourcecode(StringBuilder result, Object... parameters)
			throws Exception {
		String name = super.asSourcecode(result, parameters);
		if (grip != null) {
			TextBoostUtils.append(result, name, ".setGrip(Orientation.", grip,
					");\n");
		}

		if (up != null) {
			TextBoostUtils.append(result, name, ".setUp(", up, ");\n");
		}

		if (hasBlock != null) {
			TextBoostUtils.append(result, name, ".setHasBlock(", hasBlock,
					");\n");
		}

		if (nameOfBlock != null) {
			TextBoostUtils.append(result, name, ".setNameOfBlock(\"",
					nameOfBlock, "\");\n");
		}

		return name;
	}

}
