package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import net.sf.gluebooster.java.booster.basic.container.CountingMap;
import net.sf.gluebooster.java.booster.basic.math.ConditionBasics;
import net.sf.gluebooster.java.booster.essentials.math.Condition;
import net.sf.gluebooster.java.booster.essentials.meta.Codable;
import net.sf.gluebooster.java.booster.essentials.meta.Stringable;
import net.sf.gluebooster.java.booster.essentials.utils.MathBoostUtils;
import net.sf.gluebooster.java.booster.essentials.utils.TextBoostUtils;

/**
 * A table with blocks and hands.
 * 
 * @author CBauer
 *
 */
public class Table extends ConditionBasics<Table> implements Stringable, Codable {

	/**
	 * The extension in x-direction.
	 */
	private int max_x;
	/**
	 * The extension in y-direction.
	 */
	private int max_y;

	// private Set<Agent> agents;
	/**
	 * Set of Hand and Block.
	 */
	private Set<TableElement> tableElements;

	// public Set<Agent> getAgents() {
	// return agents;
	// }
	//
	// public void setAgents(Set<Agent> agents) {
	// this.agents = agents;
	// }

	public Table() {
		setTableElements(new HashSet<TableElement>());

	}

	/**
	 * Constructor with values
	 * 
	 * @param width
	 *            the new width (x-axis)
	 * @param depth
	 *            the new depth (y-axis)
	 * @param blockAndHands
	 *            elements of the table
	 */
	public Table(int width, int depth, /* Set<Agent> agents, */
			Set<TableElement> blockAndHands) {
		max_x = width;
		max_y = depth;
		// setAgents(agents);
		setTableElements(blockAndHands);
	}

	/**
	 * Constructor with values
	 * 
	 * @param width
	 *            the new width (x-axis)
	 * @param depth
	 *            the new depth (y-axis)
	 * @param elements
	 *            elements of the table
	 */
	public Table(int width, int depth, TableElement... elements) {
		max_x = width;
		max_y = depth;
		// setAgents(agents);
		setTableElements(elements);
	}

	@Override
	public String toString() {
		return "" + tableElements;
	}

	@Override
	public String toString(int level) {
		if (level <= 0)
			return "" + tableElements;
		else if (level == 1)
			return "Table (" + max_x + "," + max_y + ") " /* + agents */+ " "
					+ tableElements;
		else
			return getGuiString();
	}

	public int getMax_x() {
		return max_x;
	}

	public void setMax_x(int max_x) {
		this.max_x = max_x;
	}

	public int getMax_y() {
		return max_y;
	}

	public void setMax_y(int max_y) {
		this.max_y = max_y;
	}


	/**
	 * Create an extended technical name, that can be used as key in maps. The name includes the classname, to avoid naming problems with other elements of the
	 * same name, but different classes.
	 * 
	 * @param element
	 *            the element for which a technical name is to be created
	 * @return the created name
	 */
	private String createTechnicalName(TableElement<?> element) {
		return element.getClass().getSimpleName() + ":"
				+ element.getName();

	}

	@Override
	public Collection impliesImplementation(Table condition2) throws Exception {
		//elements with same name must not be contradicting

		// create a map with the names of the table elements as keys
		HashMap<String, List<TableElement<?>>> elementsMap = new HashMap<String, List<TableElement<?>>>();
		for (TableElement<?> element : getTableElements()) {
			String name = createTechnicalName(element);
			List<TableElement<?>> list = elementsMap.get(name);
			if (list == null){
				list = new ArrayList<TableElement<?>>();
				elementsMap.put(name, list);
			}
			list.add(element);
		}

		for (TableElement<?> element : condition2.getTableElements()) {
			String name = createTechnicalName(element);

			// check that there are no contradicting elements with the same name
			// and the element must be present
			if (elementsMap.containsKey(name)){
				List<TableElement<?>> elements = elementsMap.get(name);
				if (elements.isEmpty()){
					return Arrays.asList("elements are empty");
				} else {
					// if (elements.size() > 1) {
					// getLog().info("size > 1");
					// }
					for (TableElement<?> elem : elements) {
						Collection value = elem.implies(element);
						if (value == null || !value.isEmpty()) {
							return value;
						} else {
							// continue with the next element
						}
					}
				}
			} else {

				// special cases
				if (element instanceof FreeBlock) {
					String blockname = element.getName();
					for (Block block : getBlocks()) {
						if (Boolean.TRUE.equals(block.isOverBlock(blockname)))
							return Arrays.asList("block.isOverBlock");
						// a contradiction is found
					}
				} else if (element instanceof FreePlace) {
					FreePlace freePlace = (FreePlace) element;
					
					if (getOccupiedPlaces().contains(new Place(freePlace))) {
						return Arrays.asList("place is occupied");
					} // else {
						//the place is free
					//}
					
					
				} else {

					return Arrays.asList("No element " + name + " in this table. Therefore the condition is not implied");
				}
			}


		}
		
		return Collections.EMPTY_LIST;
	}


	/**
	 * A table is an elementary condition, if it has at most one block or hand
	 * predicate.
	 */
	@Override
	public boolean isElementary() {
		return tableElements.size() <= 1;
	}

	/**
	 * Creates tables with exactly one table element
	 */
	@Override
	public Set<Table> split() {
		HashSet<Table> result = new HashSet<Table>();
		if (tableElements.isEmpty()) {
			result.add(this);
		} else {
			for (TableElement element: tableElements){
				result.add(new Table(max_x, max_y, element));
			}
		}

		return result;
	}



	public Set<TableElement> getTableElements() {
		return tableElements;
	}

	public void setTableElements(Set<TableElement> blockAndHandpredicates) {
		this.tableElements = blockAndHandpredicates;
	}

	public void setTableElements(TableElement... blockAndHandpredicates) {
		tableElements = new HashSet<TableElement>();
		for (TableElement element : blockAndHandpredicates) {
			tableElements.add(element);
		}
	}

	/**
	 * Is a predicate concerning a block among the table elements.
	 * 
	 * @return true if such a predicate is found
	 */
	public boolean hasBlockPredicate() {
		for (TableElement blockOrHand : tableElements) {
			if (blockOrHand instanceof Block)
				return true;
		}

		return false;
	}

	/**
	 * Is a predicate concerning a hand among the table elements.
	 * 
	 * @return true if such a predicate is found
	 */
	public boolean hasHandPredicate() {
		for (TableElement blockOrHand : tableElements) {
			if (blockOrHand instanceof Hand)
				return true;
		}

		return false;
	}

	/**
	 * removes one element from the table
	 * 
	 * @param element
	 *            the element to remove
	 * @return true if found and removed
	 */
	public boolean removeTableElement(TableElement<?> element) {
		Object elementName = element.getName();
		Set<? extends TableElement> existing = getTableElements(element
				.getClass());
		TableElement<?> found = null;
		for (TableElement<?> x : existing) {
			if (x.getName().equals(elementName)) {
				found = x;
				break;
			}
		}

		if (found != null) {
			return tableElements.remove(found);
		} else {
			return false;
		}

	}

	/**
	 * Technical method, that should be invoked after the hash code of an element changes. Otherwise the element is no longer found in the table elements.
	 * 
	 * @param element
	 *            the element that is to be updated
	 * @return this
	 */
	public Table replaceTableElement(TableElement<?> element) throws Exception {
		removeTableElement(element);
		tableElements.add(element);
		return this;
	}

	/**
	 * Adds a new element to this table.
	 * 
	 * @param element
	 *            the new element
	 * @return this
	 */
	public Table addTableElement(TableElement<?> element) throws Exception {

		// try to blend the element with an existing element
		// example Hand 1 X=null, Y=1 and Hand 1 X=2 and Y=null should be
		// combined to Hand 1 X=2, Y=1");
		Object elementName = element.getName();
		if (elementName == null) {
			// getLog().debug("Table element has no name ", element);
			// may be ok, so no logging and no exception throw
		} else {
			for (TableElement tableElement : getTableElements(element
					.getClass())) {
				Object tableElementName = tableElement.getName();
				if (tableElementName != null) {
					if (tableElementName.equals(elementName)) {
						tableElement.add(element);
						return this;
					}
				}
			}
		}
		// if the element has not been handled yet
		tableElements.add(element);

		return this;
	}

	/**
	 * Adds multiple elements
	 * 
	 * @param elements
	 *            the elements to be added
	 */
	public void addTableElements(Collection<? extends TableElement<?>> elements)
			throws Exception {
		for (TableElement<?> element : elements)
			addTableElement(element);

	}

	@Override
	public Table createEmptyCondition() {

		return new Table(max_x, max_y);
	}

	/**
	 * Adds free places where the block will be placed
	 * 
	 * @param block
	 *            contains the place information
	 * @return this
	 */
	public Table addFreePlaces(Block block) throws Exception {
		for (Place place : block.getOccupiedPlaces()) {
			addTableElement(new FreePlace(place));
		}
		return this;
	}

	/**
	 * Gets all free places of the table
	 * 
	 * @return the found predicates
	 */
	public Collection<FreePlace> getFreePlaces() {
		return getTableElements(FreePlace.class);
	}

	/**
	 * Gets all places that are occupied by a block.
	 * 
	 * @return the found places
	 */
	public Collection<Place> getOccupiedPlaces() {
		HashSet<Place> result = new HashSet<Place>();

		for (Block block : getBlocks()) {
			result.addAll(block.getOccupiedPlaces());
		}

		return result;
	}

	/**
	 * Gets all FreeBlock-Elements. There is no computation of the elements. So in this table may be other block that are free, too.
	 * 
	 * @return the found blocks
	 */
	public Collection<FreeBlock> getFreeBlocks() {
		return getTableElements(FreeBlock.class);
	}

	/**
	 * Returns only the table elements of a given type/class.
	 * 
	 * @param ofType
	 *            the type which is wanted
	 * @return elements that are instances of subclasses of the type
	 */
	private <Type extends TableElement> Set<Type> getTableElements(
			Class<Type> ofType) {
		Set<Type> result = new HashSet<Type>();
		for (TableElement element : tableElements) {
			if (ofType.isAssignableFrom(element.getClass())) {
				result.add((Type) element);
			}
		}

		return result;
	}

	/**
	 * Finds the bottommost blocks for some places
	 * 
	 * @param places
	 *            the inspected places null places equals all places
	 * @return the found blocks
	 */
	public Set<Block> getBottommostBlocks(Collection<? extends Place> places) {
		HashSet<Block> result = new HashSet<Block>();

		for (Block block : getBlocks()) {
			if ((places == null || block.occupies(places))
					&& !Boolean.TRUE.equals(block.isOverBlocks())) {
				// it may be unknown (null) if block.isOveBlocks
				result.add(block);
			}
		}

		return result;

	}

	/**
	 * Gets the bottom most blocks of the whole table.
	 * 
	 * @return the found blocks.
	 */
	public Set<Block> getBottommostBlocks() {
		return getBottommostBlocks(null);
	}


	/**
	 * Gets the free blocks that are at the top of given places
	 * 
	 * @param places
	 *            the places to inspect
	 * @return the found blocks
	 */
	public Collection<Block> getTopmostFreeBlocks(
			Collection<? extends Place> places) {
		HashMap<Object, Block> result = new HashMap<Object, Block>();

		// first add all blocks
		for (Block block : getBlocks()) {
			result.put(block.getName(), block);
		}

		// then remove the blocks that are under other blocks
		for (Block block : getBlocks()) {
			if (Boolean.TRUE.equals(block.isOverBlocks())) {
				for (Object name : block.getOverBlocks()) {
					result.remove(name);
				}
			}
		}

		// if the places is not empty, keep only the blocks that occupy the
		// places
		for (Block block : getBlocks()) {
			if (!block.occupies(places)) {
				result.remove(block.getName());
			}
		}

		return result.values();

	}

	/**
	 * Adds forbidden places conditions.
	 * 
	 * @param places
	 *            the places to be forbidden
	 * @return the created conditions
	 */
	public Set<ForbiddenPlace> createForbiddenPlaces(
			Collection<? extends Place> places) {
		HashSet<ForbiddenPlace> result = new HashSet<ForbiddenPlace>();

		for (Place place : places) {
			result.add(new ForbiddenPlace(place));
		}

		return result;

	}

	/**
	 * Gets the hands of this table
	 * 
	 * @return the found hands
	 */
	public Set<Hand> getHands() {
		return getTableElements(Hand.class);
	}

	/**
	 * Gets the blocks of this table
	 * 
	 * @return the found blocks
	 */
	public Set<Block> getBlocks() {
		return getTableElements(Block.class);
	}

	/**
	 * Finds a block with a given name.
	 * 
	 * @param name
	 *            the name of the block
	 * @return null if nothing is found
	 */
	public Block getBlock(Object name) {
		for (Block block : getBlocks()) {
			if (block.getName().equals(name))
				return block;
		}

		return null;
	}

	/**
	 * Finds a hand with a given name.
	 * 
	 * @param name
	 *            the name of the hand
	 * @return null if nothing is found
	 */
	public Hand getHand(Object name) {
		for (Hand hand : getHands()) {
			if (hand.getName().equals(name))
				return hand;
		}

		return null;
	}

	/**
	 * Gets all blocks that lie directly over given blocks
	 * 
	 * @param blocks
	 *            the inspected blocks
	 * @return the found blocks
	 */
	public Set<Block> getBlocksDirectlyOverBlocks(
			Collection<? extends Block> blocks) {

		HashSet<Block> result = new HashSet<Block>();

		for (Block block : getBlocks()) {
			for (Block specifiedBlock : blocks) {
				if (block.getOverBlocks().contains(specifiedBlock.getName())) {
					result.add(block);
				}
			}
		}

		return result;
	}

	/**
	 * Gets the name of the block a hand is holding
	 * 
	 * @param nameOfHand
	 *            the name of the hand
	 * @return null if the hand does not hold a block
	 */
	public Object getNameOfBlockOfHandIfPossible(Object nameOfHand) {
		for (Hand hand : getHands()) {
			if (nameOfHand.equals(hand.getName())) {
				return hand.getNameOfBlock();
			}
		}
		return null;
	}

	/**
	 * Is a given block free (no block on top of this block)?
	 * 
	 * @param nameOfBlock
	 *            the inspected block
	 * @return true if no other block is on top of the block
	 */
	public boolean isBlockFree(Object nameOfBlock) {
		for (Block block : getBlocks()) {
			if (block.getOverBlocks().contains(nameOfBlock))
				return false;
		}

		return true;
	}

	@Override
	public void replaceObjectName(Object oldName, Object newName)
			throws Exception {
		// Class<TableElement> objectClass = (Class<TableElement>) object
		// .getClass();
		// for (TableElement boostedObject : getTableElements(objectClass)) {
		for (TableElement boostedObject : getTableElements()) {
			if (oldName.equals(boostedObject.getName())) {
				boostedObject.setName(newName);
			}
		}
	}

	/**
	 * TODO refactor into BoostedObject
	 * 
	 */
	@Override
	public Table cloneMe() throws Exception {
		Table result = (Table) clone();
		result.tableElements = new HashSet<TableElement>();
		for (TableElement<?> element : tableElements) {
			result.tableElements.add((TableElement) element.cloneMe());
		}
		return result;
	}

	@Override
	public void add(Condition... conditions) throws Exception {
		for (Condition condition : conditions) {
			if (condition instanceof Table) {
				addTableElements((Collection/* <? extends TableElement<?>> */) ((Table) condition).getTableElements());
			} else {
				// addTableElement((TableElement<?>) tableElement);
				throw new IllegalStateException("add element not yet supported: " + condition);
			}
		}

	}

	@Override
	public <Result extends Condition> Result findSubcondition(Result template)
			throws Exception {

		Object name = ((TableElement) template).getName();
		for (TableElement element : getTableElements((Class<TableElement>) template
				.getClass())) {
			if (name.equals(element.getName())) {
				return (Result) element;
			}
		}

		return null;
	}

	/**
	 * Finds the hand that is holding a block
	 * 
	 * @param blockTemplate
	 *            information about the block
	 * @return null if no hand is found
	 */
	public Hand getHandHoldingBlock(Block blockTemplate) {
		for (Hand hand : getHands()) {
			if (blockTemplate.getName().equals(hand.getNameOfBlock())) {
				return hand;
			}
		}

		return null;
	}

	@Override
	public Collection isInConflict(Condition condition) throws Exception {
		if (!(condition instanceof Table)) {
			return null;
		} else {
			Table table = (Table) condition;
			if (max_x != table.max_x)
				return Arrays.asList("max_x different");
			if (max_y != table.max_y)
				return Arrays.asList("max_y different");
			// if (blockcount != table.blockcount) return TRUE ;
			// if ( agents != table.agents) return FALSE; set != not yet
			// implemented
			for (TableElement element : table.tableElements) {
				for (TableElement myElement : tableElements) {
					Collection part = myElement.isInConflict(element);
					if (part != null && !part.isEmpty()) {
						return part;
					}
				}
			}

			return Collections.EMPTY_LIST;
		}

	}

			// if (element instanceof Block){
			// result |= isConflictWith((Block) element);
			// } else if (element instanceof Hand){
			// result |= isConflictWith((Hand) element);
			// } else if (element instanceof FreePlace){
			// result |= isConflictWith((FreePlace) element);
			// // } else if (element instanceof HandHasBlock ,
			// DON_T_MOVE_BLOCK){
			// // result |= isConflictWith((HandHasBlock) element);
			// } else if (element instanceof ForbiddenPlace){
			// result |= isConflictWith((ForbiddenPlace) element);
			// } else if (element instanceof FreeBlock){
			// result |= isConflictWith((FreeBlock) element);
			// } else {
			// throw new IllegalStateException("element not yet supported " +
			// element.getClass().getName());
			// }
			
			// if (result){
			// return true;
			// }

	/**
	 * Checkes that all states are known and no unknown elements are present
	 * 
	 */
	public void checkThatAllStatesAreKnown() throws IllegalStateException {
		for (TableElement element : tableElements) {
			element.checkThatAllStatesAreKnown();
		}
	}

	/**
	 * Checkes that all table elements are unique
	 * 
	 */
	public void checkSingleTableElements() throws IllegalStateException {
		CountingMap<String> counter = new CountingMap<String>();
		for (TableElement element : tableElements) {
			counter.increment(element.getClass().getSimpleName() + ": "
					+ element.getName());
		}

		for (Map.Entry<String, Long> entry : counter.entrySet()) {
			if (entry.getValue().longValue() > 1) {
				throw new IllegalStateException("Mulitiple occurences of "
						+ entry.getKey() + " in " + this);
			}
		}
	}

	@Override
	public String asSourcecode(StringBuilder result, Object... parameters)
			throws Exception {
		String uid = MathBoostUtils.getTemporaryId();
		String objectname = "table" + uid;
		TextBoostUtils.append(result, "\nTable ", objectname, "= new Table(",
				max_x, ", ", max_y, ", new HashSet());\n");

		if (tableElements != null) {
			for (TableElement element : tableElements) {
				String name = ((Codable) element).asSourcecode(result);
				TextBoostUtils.append(result, objectname,
						".getTableElements().add(", name, ");\n");
			}
		}
		return objectname;
	}

	/**
	 * A graphical string representation
	 * 
	 * @return the created string
	 */
	public String getGuiString(){
		// current precondition: the name contains one digit


		StringBuilder result = new StringBuilder("\n");

		// list the name and the position of the objects
		List<List<?>> nameXyz;

		result.append("Hands\n");
		nameXyz = new ArrayList<List<?>>();
		for (Hand hand : getHands()) {
			String name = Boolean.TRUE.equals(hand.getUp()) ? "U" : "D";
			// us "up" or "down" as name
			nameXyz.add(Arrays.asList(name, hand.getX(), hand.getY(), 1));
			// z is always 1, so that the gui-Display works
		}
		result.append(guiDisplay(nameXyz, 1));

		result.append("Up Blocks\n");
		nameXyz = new ArrayList<List<?>>();
		for (Block block : getBlocks()) {
			if (Boolean.TRUE.equals(block.getUp())) {
				String blockname = block.getName();
				for (int[] xyz : block.getOccupiedPoints(0)) {
					nameXyz.add(Arrays.asList(blockname, xyz[0], xyz[1], 1));
					// z is always 1, so that the gui-Display works
				}
			}

		}
		result.append(guiDisplay(nameXyz, 1));

		
		
		
		
		
		result.append("Blocks\n");

		//
		// Mapping of the blockname to the maximum z position
		HashMap<String, Integer> maxZ = new HashMap<String, Integer>();

		nameXyz = new ArrayList<List<?>>();
		
		int z = 0;
		Set<Block> currentBlocks = getBottommostBlocks();

		while (currentBlocks != null && !currentBlocks.isEmpty()) {
			for (Block block : currentBlocks) {
				String blockname = block.getName();
				if (maxZ.containsKey(blockname)) {
					// nothing to to, block is already handled
				} else {
					int blockZ = 0;
					Set<Object> overBlocks = block.getOverBlocks();
					if (overBlocks != null && !overBlocks.isEmpty()) {
						blockZ = maxZ.get(overBlocks.iterator().next());
					}

					for (int[] xyz : block.getOccupiedPoints(blockZ + 1)) {
						nameXyz.add(Arrays.asList(blockname, xyz[0], xyz[1],
								xyz[2]));
						blockZ = Math.max(blockZ, xyz[2]);
					}
					maxZ.put(blockname, blockZ);
					z = Math.max(blockZ, z);
				}
			}

			currentBlocks = getBlocksDirectlyOverBlocks(currentBlocks);
		}

		result.append(guiDisplay(nameXyz, z));
		
		
		return result.toString();
	}

	/**
	 * Creates a string display.
	 * 
	 * @param nameXyzList
	 *            the names and their positions
	 * @param maxZ
	 *            the maximum z position
	 * @return the created string
	 */
	private String guiDisplay(List<List<?>> nameXyzList, 
			int maxZ) {
		return guiDisplayOnlyY0(nameXyzList, maxZ);
	}
	
	/**
	 * Creates a string display
	 * 
	 * @param nameXyzList
	 *            the names and their positions
	 * @param width
	 *            the width of the table
	 * @param height
	 *            the height of the table
	 * @param maxZ
	 *            must be 1 or greater
	 * @param displayXCoordinates
	 *            should the x coordinates be displayed
	 * @param displayYCoordinates
	 *            should the y coordinates be displayed
	 * @return the created string
	 */
	private String guiDisplay(List<List<?>> nameXyzList, int width, int height,
			int maxZ, boolean displayYCoordinates, boolean displayXCoordinates) {

		
		
		String[][][] array = new String[width][height][maxZ + 1];

		for (List<?> nameXyz : nameXyzList) {
			String name = (String) nameXyz.get(0);
			int x = (Integer) nameXyz.get(1);
			int y = (Integer) nameXyz.get(2);
			int z = (Integer) nameXyz.get(3);
			array[x][y][z] = name;
		}

		StringBuilder result = new StringBuilder();

		for (int z = maxZ; z > 0; z--) {

			if (displayXCoordinates || height > 1) {
				result.append("\n");
			}

			for (int y = height - 1; y >= 0; y--) {
				if (displayYCoordinates) {
					result.append(y).append("|");
				}

				for (int x = 0; x < width; x++) {
					String value = array[x][y][z];
					if (value == null) {
						value = " ";
					} else {
						value = StringUtils.right(value, 1);
					}
					result.append(value);
				}
				result.append("\n");
			}

			// lower line
			if (displayYCoordinates) {
				result.append("  ");
			}
			if (displayXCoordinates) {
				for (int x = 0; x < width; x++) {
					result.append("-");
				}
				result.append("\n");
			}
		}
		return result.toString();
	}

	/**
	 * Creates a two-dimensional string display of the row with y=0
	 * 
	 * @param nameXyzList
	 *            the names of the elements with their positions
	 * @param maxZ
	 *            the maximum z coordinate
	 * @return the created string
	 */
	private String guiDisplayOnlyY0(List<List<?>> nameXyzList, int maxZ) {
		ArrayList<List<?>> filtered = new ArrayList<List<?>>(nameXyzList.size());

		for (List<?> nameXyz : nameXyzList) {
			int y = (Integer) nameXyz.get(2);
			if (y == 0) {
				filtered.add(nameXyz);
			}
		}

		return guiDisplay(filtered, max_x + 1, 1, maxZ, false, false);

	}

}
