package net.sf.gluebooster.demos.pojo.prolog;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import org.apache.commons.lang3.text.WordUtils;

import alice.tuprolog.NoMoreSolutionException;
import alice.tuprolog.NoSolutionException;
import alice.tuprolog.Prolog;
import alice.tuprolog.SolveInfo;
import alice.tuprolog.Struct;
import alice.tuprolog.Term;
import alice.tuprolog.Theory;
import alice.tuprolog.Var;

/**
 * A prolog implementation using tu-prolog
 * 
 * @author cbauer
 *
 */
public class TuProlog extends AbstractProlog {

	private static Struct TRUE = new Struct("true");

	private Struct clauses = new Struct();

	private Prolog engine;

	@Override
	public Object var(String name) throws Exception {
		name = WordUtils.capitalize(name);
		// name = name.replaceAll("[^A-Za-z0-9_]", "");

		return new Var(name);
	}

	@Override
	public Object atom(String value) throws Exception {
		return new Struct(value);
	}

	@Override
	public Object compound(String functor, Object... terms) throws Exception {
		switch (terms.length) {
		case 0:
			return atom(functor);
		default:
			Term[] tuTerms = new Term[terms.length];
			System.arraycopy(terms, 0, tuTerms, 0, terms.length);
			return new Struct(functor, tuTerms);
		}
	}

	@Override
	public Object rule(Object defined, Object definition) throws Exception {
		return new Struct(":-", (Struct) defined, (Struct) definition);
	}

	@Override
	public Object TRUE() {
		return TRUE;
	}

	@Override
	public void add(Object factOrRule) throws Exception {
		clauses = new Struct((Struct) factOrRule, clauses);

	}

	@Override
	public void initEnd() throws Exception {
		engine = new Prolog();
		Theory t = new Theory(clauses);
		engine.addTheory(t);

	}

	@Override
	public Iterator<Map<String, Object>> getSolutionIterator(final Object compoundWithVariables, final Object... interestingVariables) throws Exception {

		return new Iterator<Map<String, Object>>() {

			private SolveInfo currentInfo = null;
			private SolveInfo nextInfo = null;

			@Override
			public boolean hasNext() {
				// Additional result: if hasNext is true then next info is always filled.

				if (nextInfo != null) {
					return nextInfo.isSuccess();
				} else {

					if (currentInfo == null) {
						nextInfo = engine.solve((Term) compoundWithVariables);
						return nextInfo.isSuccess();
					} else if (engine.hasOpenAlternatives()) {
						try {
							nextInfo = engine.solveNext();
						} catch (NoMoreSolutionException e) {
							return false;
						}
						return true;
					} else {
						return false;
					}
				}
			}

			@Override
			public Map<String, Object> next() {
				if (!hasNext()) {// fills nextInfo
					throw new NoSuchElementException();
				} else {
					currentInfo = nextInfo;
					nextInfo = null;
					HashMap<String, Object> map = new HashMap();
					for (Object var : interestingVariables) {
						String varName = ((Var) var).getName();
						try {
							map.put(varName, currentInfo.getVarValue(varName));
						} catch (NoSolutionException e) {
							throw new IllegalStateException(e);
						}
					}
					return map;
				}

			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();

			}
		};
	}

	@Override
	public Map<String, Object> getFirstSolution(Object compoundWithVariables, Object... interestingVariables) throws Exception {
		return getSolutionIterator(compoundWithVariables, interestingVariables).next();


	}

	@Override
	public Collection<Map<String, Object>> getAllSolutions(Object compoundWithVariables, Object... interestingVariables) throws Exception {
		ArrayList<Map<String, Object>> result = new ArrayList<Map<String, Object>>();

		Iterator<Map<String, Object>> iterator = getSolutionIterator(compoundWithVariables, interestingVariables);
		while (iterator.hasNext()) {
			result.add(iterator.next());
		}

		return result;
	}

	@Override
	public String getAtomValue(Object atom) throws Exception {
		return ((Struct) atom).getName();
	}


	@Override
	public Object and(Object term1, Object term2) throws Exception {
		return compound(",", term1, term2);
	}

	@Override
	public Object or(Object term1, Object term2) throws Exception {
		return compound(";", term1, term2);
	}

	@Override
	public String getInitContent() throws Exception {
		return clauses.toString();
	}

	@Override
	public String getVarname(Object var) throws Exception {

		return ((Var) var).getName();
	}

	@Override
	public Object getAddedObjects() throws Exception {

		return clauses;
	}

	@Override
	public String getAddedObjectsAsRules() throws Exception {

		return getAddedObjects().toString();
	}

	@Override
	public Object equals(Object term1, Object term2) throws Exception {

		return compound("=", term1, term2);
	}

	@Override
	public Object list(Object... elements) throws Exception {
		Struct result = new Struct();
		for (int i = elements.length - 1; i >= 0; i--) {

			result = new Struct((Term) elements[i], result);
		}

		return result;
	}

	@Override
	public Object listWith(Object head, Object tailList) {
		return new Struct((Term) head, (Term) tailList);
	}

	@Override
	public Object[] unCompound(Object compound) throws Exception {
		Struct struct = (Struct) compound;
		int arity = struct.getArity();
		Object[] result = new Object[arity + 1];
		result[0] = struct.getName();
		for (int i = 0; i < arity; i++) {
			result[1 + i] = struct.getArg(i);
		}
		return result;
	}

	@Override
	public void humanReadable(Object object, StringBuilder result) throws Exception {
		if (object instanceof Collection) {
			humanReadable(((Collection) object).iterator(), result);
		} else if (object instanceof Iterator) {
			Iterator iterator = (Iterator) object;
			while (iterator.hasNext()) {
				Object term = iterator.next();
				humanReadable(term, result);
				result.append("\r\n");
			}
		} else if (object instanceof Var) {
			result.append(((Var) object).getName());
		} else {
			Struct struct = (Struct) object;
			if (struct.isList()) {
				humanReadable(struct.listIterator(), result);

			} else {
				String name = struct.getName();
				String DEFINED = ":-";
				if (DEFINED.equals(name)) {
					Term first = struct.getTerm(0);
					humanReadable(first, result);
					result.append(" ");
					Struct second = (Struct) struct.getTerm(1);
					if (!"true".equals(second.getName())) {
						result.append(DEFINED).append(" ");
						humanReadable(second, result);
					}

				} else {
					result.append(name);
					int arity = struct.getArity();
					if (arity > 0) {
						result.append("(");
						for (int i = 0; i < arity; i++) {
							humanReadable(struct.getTerm(i), result);
							if (i < arity - 1) {
								result.append(", ");
							}
						}
						result.append(") ");
					}
				}
			}
		}
	}

}
