package net.sf.gluebooster.demos.pojo.math;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

import net.sf.gluebooster.demos.pojo.math.library.Basics;
import net.sf.gluebooster.demos.pojo.math.library.logic.Bool;
import net.sf.gluebooster.demos.pojo.math.library.logic.Logic;
import net.sf.gluebooster.demos.pojo.math.library.setTheory.ClassesSets;
import net.sf.gluebooster.demos.pojo.prolog.Prolog;
import net.sf.gluebooster.java.booster.basic.container.Tuple;
import net.sf.gluebooster.java.booster.essentials.TestRoot;

/**
 * Tests of the prolog proof generator.
 * 
 * @author cbauer
 *
 */
public class PrologProofGeneratorTest extends TestRoot {

	@Test
	public void test_1() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		List<Statement> lib = Arrays.asList(Logic.EQUALS);
		Prolog prolog = generator.createPrologWithStatements(lib);
		String rules = prolog.getAddedObjectsAsRules();

		Object a = prolog.atom("a");
		Object b = prolog.atom("b");

		String equality = generator.getFunctor(Logic.EQUALS);

		Assert.assertTrue(prolog.getSolutionIterator(prolog.compound(equality, prolog.list(a))).hasNext());
		Assert.assertTrue(prolog.getSolutionIterator(prolog.compound(equality, prolog.list(a, a))).hasNext());
		Assert.assertTrue(prolog.getSolutionIterator(prolog.compound(equality, prolog.list(a, a, a))).hasNext());
		Assert.assertFalse(prolog.getSolutionIterator(prolog.compound(equality, prolog.list(a, b))).hasNext());
		Assert.assertFalse(prolog.getSolutionIterator(prolog.compound(equality, prolog.list(a, a, b))).hasNext());

		lib = Arrays.asList(Bool.VERUM, Bool.FALSUM, Bool.NOT_RELATION, Bool.NAND_RELATION);
		prolog = generator.createPrologWithStatements(lib);
		rules = prolog.getAddedObjectsAsRules();

		String x = "X";
		Object varX = prolog.var(x);
		String notX = "NotX";
		Object varNotX = prolog.var(notX);

		Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(prolog.compound(generator.getFunctor(Bool.NOT_RELATION), varX, varNotX), varX,
				varNotX);
		Assert.assertEquals(2, allSolutions.size());
		String allSolutionsText = allSolutions.toString();
		Assert.assertTrue(allSolutionsText.contains(Bool.VERUM.getIdentifier().toString()));

		lib = Arrays.asList(Logic.EQUALS, Bool.VERUM, Bool.FALSUM, Bool.NOT_RELATION, Bool.NAND_RELATION);
		prolog = generator.createPrologWithStatements(lib);
		rules = prolog.getAddedObjectsAsRules();

		// lib = Arrays.asList(Logic.EQUALS, Boolean.VERUM, Boolean.FALSUM, Boolean.NOT_FUNCTION, Boolean.NAND_RELATION, Boolean.NOT_NAIVE, Boolean.NAND_NAIVE);
		// prolog = generator.createPrologWithStatements(lib);
		// rules = prolog.getAddedObjectsAsRules();
		// allSolutions = prolog.getAllSolutions(prolog.compound(generator.getFunctor(Boolean.NOT_NAIVE), varX, varNotX), varX, varNotX);
		// Assert.assertEquals(2, allSolutions.size());

		// lib = Arrays.asList(Logic.EQUALS, Boolean.VERUM, Boolean.FALSUM, Boolean.NOT_RELATION, Boolean.NAND_RELATION, Boolean.NOT_NAIVE, Classes.ELEMENT_OF,
		// Boolean.NAND_NAIVE, Boolean.NOT_FROM_OTHERS);
		// prolog = generator.createPrologWithStatements(lib);
		// rules = prolog.getAddedObjectsAsRules();
	}

	@Test
	@Ignore("no longer working")
	public void test0() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		List<Statement> lib = Arrays.asList(Bool.VERUM, Bool.FALSUM, Bool.BOOLEAN_SET, ClassesSets.ELEMENT_OF);
		Prolog prolog = generator.createPrologWithStatements(lib);

		String added = prolog.getAddedObjects().toString();
		// System.out.println(added);
		String booleanFunctor = generator.getFunctor(Bool.BOOLEAN_SET);
		Object varX = prolog.var();
		String x = prolog.getVarname(varX);

		Collection<Map<String, Object>> solutions = prolog.getAllSolutions(prolog.compound(booleanFunctor, varX), varX);

		Assert.assertEquals(2, solutions.size());// true and false

		String verum = generator.getFunctor(Bool.VERUM);
		String falsum = generator.getFunctor(Bool.FALSUM);
		List verumFalsum = new ArrayList<>(Arrays.asList(verum, falsum));
		Iterator<Map<String, Object>> iterator = solutions.iterator();
		while (iterator.hasNext()) {
			verumFalsum.remove(prolog.getAtomValue(iterator.next().get(x)));
		}
		Assert.assertEquals(0, verumFalsum.size());

	}

	@Test
	@Ignore("no longer working")
	public void test1() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		Prolog prolog = generator.createPrologWithStatements(MathStudiesProofs.getStatementsUpTo(Bool.NOT_FROM_OTHERS));

		String booleanFunctor = generator.getFunctor(Bool.BOOLEAN_SET);

		Object varX = prolog.var();
		String x = prolog.getVarname(varX);

		Collection<Map<String, Object>> solutions = prolog.getAllSolutions(prolog.compound(booleanFunctor, varX), varX);

		Assert.assertEquals(2, solutions.size());// true and false
		
		String verum = generator.getFunctor(Bool.VERUM);
		String falsum = generator.getFunctor(Bool.FALSUM);

		List verumFalsum = new ArrayList<>(Arrays.asList(verum, falsum));
		Iterator<Map<String, Object>> iterator = solutions.iterator();
		while (iterator.hasNext()) {
			verumFalsum.remove(prolog.getAtomValue(iterator.next().get(x)));
		}
		Assert.assertEquals(0, verumFalsum.size());
		
	}


	@Test
	public void test2a() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		List<Statement> lib = Arrays.asList(Bool.VERUM, Bool.FALSUM, Bool.BOOLEAN_SET, Bool.NOT_RELATION, ClassesSets.ELEMENT_OF);

		Statement varA = Logic.var("A");
		// Statement aIsBoolean = Classes.isElementOf(varA, Boolean.BOOLEAN_SET);
		Statement notA = Bool.not(varA);

		// Statement valueTable = generator.displayValueTable(lib, new Statement[] { aIsBoolean, notA }, varA, notA);
		Statement valueTable = generator.displayValueTable(lib, new Statement[] {}, varA, notA);
		Assert.assertNotNull(valueTable);
		Object[] rows = valueTable.getRawData();
		Assert.assertEquals(3, rows.length);// header + 2 values
	}

	@Test
	public void test2b() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		Statement varA = Logic.var("A");
		// Statement aIsBoolean = Classes.isElementOf(varA, Boolean.BOOLEAN_SET);
		Statement notA = Bool.not(varA);
		Statement aNandA = Bool.nand(varA, varA);

		// Statement valueTable = generator.displayValueTable(MathStudiesProofs.getStatementsUpTo(Boolean.NOT_FROM_OTHERS), new Statement[] {}, varA, varA,
		// Boolean.nor(varA, varA),
		// notA);
		Statement valueTable = generator.displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NOT_FROM_OTHERS), new Statement[] {}, varA, varA, aNandA,
				notA);
		Assert.assertNotNull(valueTable);
		Object[] rows = valueTable.getRawData();
		Assert.assertEquals(3, rows.length);// header + 2 rows for A=true and A=false
	}

	@Test
	@Ignore("no longer working")
	// @Ignore("not yet functional")
	public void test3() throws Exception {

		PrologProofGenerator generator = new PrologProofGenerator();

		List<Statement> lib = Arrays.asList(Bool.VERUM, Bool.FALSUM, Bool.BOOLEAN_SET, ClassesSets.ELEMENT_OF);
		// Prolog prolog = generator.createPrologWithStatements(lib);

		Statement varA = Logic.var("A");
		Statement aIsBoolean = ClassesSets.elementOf(varA, Bool.BOOLEAN_SET);

		Tuple<Prolog, Object, Object[], Map<Statement, Object>> prologEnv = generator
				.createPrologAndQuery(MathStudiesProofs.getStatementsUpTo(Bool.NOT_FROM_OTHERS), new Statement[] { aIsBoolean }, varA);

		Prolog prolog = prologEnv.getFirst();

		StringBuilder text = new StringBuilder();
		prolog.humanReadable(prolog.getAddedObjects(), text);
		text.append("\r\nthe query\r\n");
		prolog.humanReadable(prologEnv.getSecond(), text);
		getLog().info(text);

		Statement valueTable = generator.displayValueTable(MathStudiesProofs.getStatementsUpTo(Bool.NOT_FROM_OTHERS), new Statement[] { aIsBoolean }, varA);
		Assert.assertNotNull(valueTable);
		if (Basics.TABLE.is(valueTable)) {
			Assert.assertEquals(3, ((Object[]) valueTable.getRawData()).length); // header + true + false
		}
	}

	// @Test
	// public void computeImplicationProofTest() throws Exception {
	// PrologProofGenerator generator = new PrologProofGenerator();
	// Statement A = Logic.var("A");
	// Statement B = Logic.var("B");
	// Statement C = Logic.var("C");
	//
	// Statement proof = generator.computeImplicationProof(MathStudiesProofs.getStatementsUpTo(Boolean.BICONDITIONAL_FROM_OTHERS),
	// Logic.bracket(Boolean.and(Boolean.andWithBrackets(Boolean.implies(A, B), Boolean.implies(B, C)), Logic.bracket(Boolean.implies(C, A)))),
	// Boolean.implies(A, C));
	// Assert.assertNotNull(proof);
	// }

	/**
	 * Tests that prolog is created correctly
	 * 
	 * @throws Exception
	 */
	@Test
	@Ignore("no longer working")
	// @Ignore("prolog not yet finished. Too complex")
	public void testPrologGeneration() throws Exception {
		PrologProofGenerator generator = new PrologProofGenerator();

		Pair<String, String> dataAndQuery = generator.createPrologForProof(MathStudiesProofs.getStatementsUpTo(Bool.RINGSCHLUSS), Bool.RINGSCHLUSS);

		Assert.assertNotNull(dataAndQuery);

		getLog().info("data and query\n", dataAndQuery);

	}

}
