package net.sf.gluebooster.demos.pojo.planning.blockworld;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.collections4.Transformer;
import org.junit.Assert;
import org.junit.Test;

import net.sf.gluebooster.java.booster.basic.container.BoostedHashSet;
import net.sf.gluebooster.java.booster.basic.container.BoostedNode;
import net.sf.gluebooster.java.booster.basic.gui.swing.JOptionPaneUserInteraction;
import net.sf.gluebooster.java.booster.basic.math.Operation;
import net.sf.gluebooster.java.booster.basic.math.graph.GraphDisplayConfiguration;
import net.sf.gluebooster.java.booster.basic.math.planning.AbstractPlanningTest;
import net.sf.gluebooster.java.booster.basic.math.planning.MetaPlanner;
import net.sf.gluebooster.java.booster.basic.math.planning.Plan;
import net.sf.gluebooster.java.booster.basic.math.planning.PlannerEnv;
import net.sf.gluebooster.java.booster.basic.math.planning.Planning;
import net.sf.gluebooster.java.booster.essentials.eventsCommands.Callable;
import net.sf.gluebooster.java.booster.essentials.logging.LogBooster;
import net.sf.gluebooster.java.booster.essentials.meta.Example;

public class BlockworldTest extends AbstractPlanningTest implements ActionListener {


	private String getDefaultBlockname(int blocknumber){

		return  "Block " + blocknumber;
	}
	
	
	private Block getDefaultBlock(int blocknumber, int x, int y,
			Orientation orientation, boolean up, int[] overBlockNumbers) {
		String blockname = getDefaultBlockname(blocknumber);
		Blocktype type;
		switch (blocknumber) {
		case 1:
			type = Blocktype.QUADRUPLEBLOCK;
			break;
		case 2:
			type = Blocktype.DOUBLEBLOCK;
			break;
		default:
			type = Blocktype.SINGLEBLOCK;
			break;
		}
		HashSet<Object> overBlocks = new HashSet<Object>();
		if (overBlockNumbers != null) {
			for (int overBlock : overBlockNumbers) {
				overBlocks.add(getDefaultBlockname(overBlock));
			}
		}

		return new Block(blockname, x, y, type, orientation, up, overBlocks);

	}
	
	private Hand getDefaultHand() {
		Hand result = new Hand("Hand 1", 0, 0);
		result.setUp(Boolean.TRUE);
		return result;
	}

	private Table getDefaultTable(Set<TableElement> blocksAndHands) {
		int width = 7;
		int depth = 5;
		return new Table(width, depth, /* agents, */blocksAndHands);

	}

	/**
	 * Creates a pyramid of blocks
	 */
	private Planning<Table> createPyramidTestdata(int numberOfBlocks,
			int stepsToRun)
			throws Exception {


		Set<TableElement> initialBlocksAndHands = new HashSet<TableElement>();
		Set<TableElement> wantedBlocksAndHands = new HashSet<TableElement>();

		initialBlocksAndHands.add(getDefaultHand());

		for (int i = 1; i <= numberOfBlocks; i++) {

			int[] overBlocks = null;
			if (i > 1) {
				overBlocks = new int[] { (i - 1) };
			}

			initialBlocksAndHands.add(getDefaultBlock(i, 0, 0, Orientation.Y,
					false, overBlocks));
			wantedBlocksAndHands.add(getDefaultBlock(i, 2, 0, Orientation.Y,
					false, overBlocks));

		}

		Plan<Table> plan = new Plan<Table>(new Blockworld(),
				getDefaultTable(initialBlocksAndHands),
				getDefaultTable(wantedBlocksAndHands));

		PlannerEnv env = new PlannerEnv();
		env.setPlanningLog(new LogBooster("planner"));
		env.setMaxSteps(stepsToRun);


		return new Planning<Table>(plan, env);
	}

	@Test
	public void testIdlePlanner() throws Exception {

		Planning<Table> data = createPyramidTestdata(3, 100);

		Callable<Planning<Table>, Planning<Table>> planner = MetaPlanner.createIdlePlanner();
		planner.call(data);

	}

	/**
	 * Tests that the planner recognizes a situation that is already solved.
	 * 
	 * @throws Exception
	 */
	@Test
	public void testNoNeedToDoAnything() throws Exception {
		Planning<Table> data = createPyramidTestdata(3, 1);
		// make the postcondition equal to the precondition (only without the hand)
		Operation<Table> riddle = data.getPlan().getOperation(
				data.getPlan().getUnsolvedNodes().iterator().next());
		Table condition = riddle.getGlobalPrecondition().cloneMe();
		Hand hand = condition.getHands().iterator().next();
		condition.removeTableElement(hand);
		riddle.setPostcondition(condition);
		
		Callable<Planning<Table>, Planning<Table>> planner = MetaPlanner.createPlanner1();
		
		for (int i = 0; i <100; i++) {
			data = planner.call(data);// do some steps
		}
		
		data.getPlan().simpleDisplay();

		Assert.assertTrue("The plan should be solved.", data.getPlan().isSolved());
		Assert.assertEquals(
				"There should be only one node in the plan, because nothing is to do",
				1, data.getPlan().getGraph().getAllVertices().size());
		
	}

	@Test
	public void testPlanner1() throws Exception {
		testPlanner(MetaPlanner.createPlanner1(), 5, 30);
	}

	private Planning<Table> createPyramidRiddle(int numberOfBlocks, StringBuilder operationsLog, StringBuilder resultLog) throws Exception {

		Planning<Table> data = createPyramidTestdata(numberOfBlocks, 1);
		data.setName("Pyramid-Riddle with " + numberOfBlocks + " blocks");

		JOptionPaneUserInteraction interaction = new JOptionPaneUserInteraction();

		GraphDisplayConfiguration<BoostedNode, Object> displayConfiguration = data.getPlan().createSolveStateDisplayConfiguration();
		data.getPlan().setDefaultDisplayConfiguration(displayConfiguration);
		data.simpleDisplay();
		BoostedHashSet<BoostedNode> nodes = data.getPlan().getGraph().getSomeNodesOfTheConnectedComponents();
		nodes.setInformListenerWhenEmpty(true);
		// nodes.getListeners().addActionListener(this);
		nodes.getListeners().addDefaultHandler(this);
		BoostedNode riddle = nodes.iterator().next();

		String riddleString = "Riddle\n" + data.getPlan().getOperation(riddle).toString(3) + "\n=============\n\n";

		if (operationsLog != null) {
			operationsLog.append(riddleString);
		}

		if (resultLog != null) {
			resultLog.append(riddleString);
		}

		return data;
	}

	@Test
	@Example(clasz = Blockworld.class)
	public void pyramidDemo() throws Exception {

		StringBuilder resultLog = new StringBuilder();
		// [gb:useInDocumentation]
		// An example how the planner relocates a pyramid.
		Planning<Table> data = createPyramidRiddle(3, null, resultLog);
		solve(data, (Callable) MetaPlanner.createPlanner1(), 1000, 1000, null, resultLog, null, null, "pyramidDemo");
		// [/gb:useInDocumentation]

		data.getPlan().logOperatorsAndDetails(resultLog);
		writeExampleResult(resultLog);


	}

	private void testPlanner(Callable thePlanner,
			int firstStepsWithoutDisplay, int stepsWithDisplay)
			throws Exception {

		int minNumberOfBlocks = 3;
		int maxNumberOfBlocks = 3;


		for (int numberOfBlocks = minNumberOfBlocks; numberOfBlocks <= maxNumberOfBlocks; numberOfBlocks++) {

			StringBuilder operations = new StringBuilder();
			StringBuilder result = new StringBuilder();

			Planning<Table> data = createPyramidRiddle(numberOfBlocks, operations, result);

			solve(data, thePlanner, firstStepsWithoutDisplay, stepsWithDisplay, operations, result, null, null, "testPlanner");

		}
	}

	@Test
	public void testPlannerThatJustExpands() throws Exception {
		testPlanner(MetaPlanner.createPlannerThatJustExpands(), 18, 40);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		getLog().info("actionPerformed", e);

	}

	@Test
	public void testTableGuiString() throws Exception {
		Planning<Table> planning = createPyramidTestdata(3, 1);
		Operation<Table> operation = planning.getPlan().getOperation(
				planning.getPlan().getGraph().getAllVertices().iterator()
						.next());

		String gui = operation.getGlobalPrecondition().getGuiString();
		Assert.assertNotNull(gui);
		getLog().info(gui);
	}

}
