package net.sf.gluebooster.demos.pojo.prolog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.junit.Ignore;
import org.junit.Test;

import alice.tuprolog.SolveInfo;
import alice.tuprolog.Struct;
import alice.tuprolog.Theory;
import alice.tuprolog.Var;
import junit.framework.Assert;

/**
 * Online Prolog: http://swish.swi-prolog.org/
 * 
 * @author cbauer
 *
 */
public class PrologTest {

	private List<Prolog> getPrologEngines() {
		return Arrays.asList((Prolog) new TuProlog());
	}


	@Test
	public void test1() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			String predicate = "p";
			String one = "1";
			String two = "2";

			Object atom1 = prolog.atom(one);
			Object atom2 = prolog.atom(two);
			Object statement1 = prolog.compound(predicate, atom1);
			Object statement2 = prolog.compound(predicate, atom2);
			prolog.add(prolog.fact(statement1));
			prolog.add(prolog.fact(statement2));

			prolog.initEnd();

			String x = "X";
			Object varX = prolog.var(x);
			Map<String, Object> firstSolution = prolog.getFirstSolution(prolog.compound(predicate, varX), varX);
			Assert.assertNotNull(firstSolution);
			Assert.assertTrue(firstSolution.containsKey(x));
			String value = prolog.getAtomValue(firstSolution.get(x));
			Assert.assertTrue(one.equals(value) || two.equals(value));

			Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(prolog.compound(predicate, varX), varX);
			Assert.assertEquals(2, allSolutions.size());
			List oneTwo = new ArrayList<>(Arrays.asList(one, two));
			Iterator<Map<String, Object>> iterator = allSolutions.iterator();
			while (iterator.hasNext()) {
				oneTwo.remove(prolog.getAtomValue(iterator.next().get(x)));
			}
			Assert.assertEquals(0, oneTwo.size());

		}
	}

	@Test
	public void test2() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			String VERUM = "verum";
			String FALSUM = "falsum";
			String BOOLEAN = "Boolean";
			String NOT = "not";
			String NAND = "nand";

			Object verum = prolog.atom(VERUM);
			Object falsum = prolog.atom(FALSUM);

			prolog.addFact(prolog.compound(BOOLEAN, verum));
			prolog.addFact(prolog.compound(BOOLEAN, falsum));

			prolog.addFact(prolog.compound(NOT, falsum, verum));
			prolog.addFact(prolog.compound(NOT, verum, falsum));

			prolog.addFact(prolog.compound(NAND, verum, verum, falsum));
			prolog.addFact(prolog.compound(NAND, verum, falsum, verum));
			prolog.addFact(prolog.compound(NAND, falsum, verum, verum));
			prolog.addFact(prolog.compound(NAND, falsum, falsum, verum));

			prolog.initEnd();

			String a = "A";
			Object varA = prolog.var(a);
			String b = "B";
			Object varB = prolog.var(b);
			String notA = "NotA";
			Object varNotA = prolog.var(notA);
			String aNandB = "AnandB";
			Object varANandB = prolog.var(aNandB);

			Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(prolog.conjunction(
					/*
					 * prolog.compound(BOOLEAN, varA), prolog.compound(BOOLEAN, varB), prolog.compound(BOOLEAN, varNotA), prolog.compound(BOOLEAN, varANandB),
					 */ prolog.compound(NOT, varA, varNotA),
					prolog.compound(NAND, varA, varB, varANandB)), varA, varB, varNotA,
					varANandB);
			Assert.assertEquals(4, allSolutions.size());

		}
	}

	@Test
	public void test3() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			String VERUM = "verum";
			String FALSUM = "falsum";
			String BOOLEAN = "Boolean";

			Object verum = prolog.atom(VERUM);
			Object falsum = prolog.atom(FALSUM);

			prolog.addFact(prolog.compound(BOOLEAN, verum));
			prolog.addFact(prolog.compound(BOOLEAN, falsum));

			prolog.initEnd();

			String a = "A";
			Object varA = prolog.var(a);
			String b = "B";
			Object varB = prolog.var(b);

			Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(prolog.conjunction(prolog.equals(varA, verum), prolog.equals(varB, varA)),
					varB);
			Assert.assertEquals(1, allSolutions.size());// B = verum

			allSolutions = prolog.getAllSolutions(prolog.conjunction(prolog.equals(varA, verum), prolog.equals(varB, falsum), prolog.equals(varB, varA)), varB);
			Assert.assertEquals(0, allSolutions.size());// No solution

		}
	}

	@Test
	public void test4() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			String VERUM = "verum";
			String FALSUM = "falsum";
			String BOOLEAN = "Boolean";
			String NOT = "not";
			String NAND = "nand";
			String AND = "and";

			Object verum = prolog.atom(VERUM);
			Object falsum = prolog.atom(FALSUM);

			prolog.addFact(prolog.compound(BOOLEAN, verum));
			prolog.addFact(prolog.compound(BOOLEAN, falsum));

			prolog.addFact(prolog.compound(NOT, falsum, verum));
			prolog.addFact(prolog.compound(NOT, verum, falsum));

			prolog.addFact(prolog.compound(NAND, verum, verum, falsum));
			prolog.addFact(prolog.compound(NAND, verum, falsum, verum));
			prolog.addFact(prolog.compound(NAND, falsum, verum, verum));
			prolog.addFact(prolog.compound(NAND, falsum, falsum, verum));

			prolog.initEnd();

			String a = "A";
			Object varA = prolog.var(a);
			String b = "B";
			Object varB = prolog.var(b);
			String notA = "NotA";
			Object varNotA = prolog.var(notA);
			String aNandB = "AnandB";
			Object varANandB = prolog.var(aNandB);

			Collection<Map<String, Object>> allSolutions = prolog.getAllSolutions(prolog.conjunction(
					/*
					 * prolog.compound(BOOLEAN, varA), prolog.compound(BOOLEAN, varB), prolog.compound(BOOLEAN, varNotA), prolog.compound(BOOLEAN, varANandB),
					 */ prolog.compound(NOT, varA, varNotA), prolog.compound(NAND, varA, varB, varANandB)), varA, varB, varNotA, varANandB);
			Assert.assertEquals(4, allSolutions.size());

		}
	}

	@Test
	public void testTuPrologLists() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			Object x = prolog.var("X");
			Object tail = prolog.var("T");
			Object any = prolog.var("A");
			prolog.addFact(prolog.compound("member", x, prolog.listWith(x, tail))); // member(X, [X|_]).
			prolog.addRule(prolog.compound("member", x, prolog.listWith(any, tail)), prolog.compound("member", x, tail)); // member(X, [_|T]) :- member(X, T).

			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());

			Object a = prolog.atom("a");
			Object b = prolog.atom("b");
			Object c = prolog.atom("c");

			Object aList = prolog.list(c, a, a);

			Object z = prolog.var("Z");

			Collection<Map<String, Object>> solution = prolog.getAllSolutions(prolog.compound("member", z, aList), z);
			Assert.assertTrue(solution.size() > 1);// z=c, z=a maybe duplicate z=a

			Assert.assertTrue(prolog.getSolutionIterator(prolog.compound("member", a, aList)).hasNext());
			Assert.assertTrue(prolog.getSolutionIterator(prolog.compound("member", c, aList)).hasNext());
			Assert.assertFalse(prolog.getSolutionIterator(prolog.compound("member", b, aList)).hasNext());
		}

	}

	@Test
	public void testTuPrologLists2() throws Exception {
		for (Prolog prolog : getPrologEngines()) {
			Object head = prolog.var("H");
			Object head2 = prolog.var("H2");
			Object tail = prolog.var("T");
			// lists with all equal elements
			prolog.addFact(prolog.compound("equalElements", prolog.list(head))); // list with one element is ok
			prolog.addRule(prolog.compound("equalElements", prolog.listWith(head, prolog.listWith(head, tail))),
					prolog.compound("equalElements", prolog.listWith(head, tail)));

			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());

			Object a = prolog.atom("a");
			Object b = prolog.atom("b");

			Assert.assertTrue(prolog.getSolutionIterator(prolog.compound("equalElements", prolog.list(a))).hasNext());
			Assert.assertTrue(prolog.getSolutionIterator(prolog.compound("equalElements", prolog.list(a, a))).hasNext());
			Assert.assertTrue(prolog.getSolutionIterator(prolog.compound("equalElements", prolog.list(a, a, a))).hasNext());
			Assert.assertFalse(prolog.getSolutionIterator(prolog.compound("equalElements", prolog.list(a, b))).hasNext());
			Assert.assertFalse(prolog.getSolutionIterator(prolog.compound("equalElements", prolog.list(a, a, b))).hasNext());

		}
	}


	@Test
	public void testProofGenerating0() throws Exception {
		for (Prolog prolog : getPrologEngines()) {

			// Facts + Rules
			// implies(a,b).
			// implies(b,a).
			// implies(X,Y):- implies(X,Z), implies(Z,Y).
			// iff(X,Y):- implies(X,Y), implies(Y,X).

			// Query iff(X, Y).

			Object a = prolog.atom("a");
			Object b = prolog.atom("b");

			Object X = prolog.var("X");
			Object Y = prolog.var("Y");

			String iff = "iff";
			String implies = "implies";


			prolog.addFact(prolog.compound(implies, a, b));
			prolog.addFact(prolog.compound(implies, b, a));
			prolog.addRule(prolog.compound(iff, X, Y), prolog.conjunction(prolog.compound(implies, X, Y), prolog.compound(implies, Y, X)));


			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());


			Object toBeProven = prolog.compound(iff, X, Y);

			Map<String, Object> theProof = prolog.getFirstSolution(toBeProven, X, Y);
			Assert.assertNotNull(theProof);


		}
	}


	/**
	 * Rule: X <=>Y = (X=>Y and Y => X)
	 * 
	 * @throws Exception
	 */
	@Test
	public void testProofGenerating1() throws Exception {

		// Facts (Preconditions)
		// implies(a,b, [proofPart1]).
		// implies(b,a, [proofPart2]).

		// Rule
		// iff(X,Y,[rule1, Proof1, Proof2]):- implies(X,Y, Proof1), implies(Y,X, Proof2).

		// Query ?- iff(X,Y,Proof).
		// result should be
		// Proof = [rule1, [proofPart1], [proofPart2]],
		// X = a,
		// Y = b

		// Test 1 facts in the prolog engine
		for (Prolog prolog : getPrologEngines()) {
			Object X = prolog.var("X");
			Object Y = prolog.var("Y");
			String proofVar = "Proof";
			Object Proof = prolog.var(proofVar);
			Object Proof1 = prolog.var("Proof1");
			Object Proof2 = prolog.var("Proof2");

			String ruleString = "rule1";
			Object rule1 = prolog.atom(ruleString);
			Object a = prolog.atom("a");
			Object b = prolog.atom("b");
			String proofPart1String = "proofPart1";
			String proofPart2String = "proofPart2";
			Object proofPart1 = prolog.atom(proofPart1String);
			Object proofPart2 = prolog.atom(proofPart2String);

			String iff = "iff";
			String implies = "implies";

			// Facts
			prolog.addFact(prolog.compound(implies, a, b, prolog.list(proofPart1)));
			prolog.addFact(prolog.compound(implies, b, a, prolog.list(proofPart2)));

			// Rule
			prolog.addRule(prolog.compound(iff, X, Y, prolog.list(rule1, Proof1, Proof2)),
					prolog.conjunction(prolog.compound("implies", X, Y, Proof1), prolog.compound("implies", Y, X, Proof2)));


			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());


			Object toBeProven = prolog.conjunction(//
					prolog.compound(iff, X, Y, Proof)
			);

			Map variables = prolog.getFirstSolution(toBeProven, Proof);
			Object theProof = variables.get(proofVar);
			Assert.assertNotNull(theProof);
			StringBuilder text = new StringBuilder();
			prolog.humanReadable(theProof, text);
			String string = text.toString();
			Assert.assertTrue(string.contains(ruleString));
			Assert.assertTrue(string.contains(proofPart1String));
			Assert.assertTrue(string.contains(proofPart2String));


		}

	}


	/**
	 * Adding transitivity rule (without proof)
	 * 
	 * @throws Exception
	 */
	@Test
	@Ignore("Does not work with tuprolog (no result, probably bad solving algorithm)")
	public void testProofGenerating2a() throws Exception {

		// ?-iff(a,c, Proof).


		// Facts (Preconditions)
		// implies(a,b).
		// implies(b,c).
		// implies(c,a).

		// Rule
		// transitivity of implies
		// implies(A,C):- implies(A,B), implies( B,C).
		// X <=>Y = (X=>Y and Y => X)
		// iff(X,Y):- implies(X,Y), implies(Y,X).

		// Query ?- iff(a,c).

		// Test 1 facts in the prolog engine
		for (Prolog prolog : getPrologEngines()) {
			Object A = prolog.var("A");
			Object B = prolog.var("B");
			Object C = prolog.var("C");

			Object a = prolog.atom("a");
			Object b = prolog.atom("b");
			Object c = prolog.atom("c");

			String iff = "iff";
			String implies = "implies";

			Object iffAtom = prolog.atom(iff);
			Object impliesAtom = prolog.atom(implies);

			// Facts
			prolog.addFact(prolog.compound(implies, a, b));
			prolog.addFact(prolog.compound(implies, b, c));
			prolog.addFact(prolog.compound(implies, c, a));

			// Rule
			prolog.addRule(prolog.compound(iff, A, B), prolog.conjunction(prolog.compound("implies", A, B), prolog.compound("implies", B, A)));

			prolog.addRule(prolog.compound(implies, A, C), prolog.conjunction(prolog.compound("implies", A, B), prolog.compound("implies", B, C)));


			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());


			Object toBeProven = prolog.conjunction(//
					prolog.compound(implies, a, A));
			Map variables = prolog.getFirstSolution(toBeProven, A);
			Object theProof = variables.get(A);
			Assert.assertNotNull(theProof);

			// toBeProven = prolog.conjunction(//
			// prolog.compound(iff, a, c, Proof));
			//
			// variables = prolog.getFirstSolution(toBeProven, Proof);
			// theProof = variables.get(proofVar);
			// Assert.assertNotNull(theProof);
			// StringBuilder text = new StringBuilder();
			// prolog.humanReadable(theProof, text);
			// String string = text.toString();
			// Assert.assertTrue(string.contains(ruleString));
			// Assert.assertTrue(string.contains(proofPart1String));
			// Assert.assertTrue(string.contains(proofPart2String));

		}


	}

	/**
	 * Adding transitivity rule (with proof)
	 * 
	 * @throws Exception
	 */
	@Test
	@Ignore("Does not work with tuprolog (no result, probably bad solving algorithm)")
	public void testProofGenerating2b() throws Exception {

		// ?-iff(a,c, Proof).

		// Facts (Preconditions)
		// implies(a,b, [implies,a,b, becauseof, proofPart1]).
		// implies(b,c, [implies,b,c, becauseof,proofPart2]).
		// implies(c,a, [implies,c,a, becauseof,proofPart3]).

		// Rule
		// transitivity of implies
		// implies(A,C, [implies,A,C, becauseof, transitiv, Proof1, Proof2]):- implies(A,B, Proof1), implies( B,C, Proof2).
		// X <=>Y = (X=>Y and Y => X)
		// iff(X,Y,[iff, X, Y, becauseOf, rule1, Proof1, Proof2]):- implies(X,Y, Proof1), implies(Y,X, Proof2).

		// Query ?- iff(a,c,Proof).
		// result should be
		// Proof = [iff, a, c, becauseOf, rule1, [implies, a, c, becauseof, transitiv, [implies, a, b, becauseof, proofPart1], [implies, b, c, becauseof,
		// proofPart2]], [implies, c, a, becauseof, proofPart3]]

		// Test 1 facts in the prolog engine
		for (Prolog prolog : getPrologEngines()) {
			Object A = prolog.var("A");
			Object B = prolog.var("B");
			Object C = prolog.var("C");
			String proofVar = "Proof";
			Object Proof = prolog.var(proofVar);
			Object Proof1 = prolog.var("Proof1");
			Object Proof2 = prolog.var("Proof2");

			String ruleString = "rule1";
			Object rule1 = prolog.atom(ruleString);
			Object a = prolog.atom("a");
			Object b = prolog.atom("b");
			Object c = prolog.atom("c");
			String proofPart1String = "proofPart1";
			String proofPart2String = "proofPart2";
			String proofPart3String = "proofPart3";
			Object proofPart1 = prolog.atom(proofPart1String);
			Object proofPart2 = prolog.atom(proofPart2String);
			Object proofPart3 = prolog.atom(proofPart3String);

			String iff = "iff";
			String implies = "implies";
			String becauseOf = "becauseOf";
			String transitive = "transitive";

			Object iffAtom = prolog.atom(iff);
			Object impliesAtom = prolog.atom(implies);
			Object becauseOfAtom = prolog.atom(becauseOf);
			Object transitiveAtom = prolog.atom(transitive);

			// Facts
			prolog.addFact(prolog.compound(implies, a, b, prolog.list(impliesAtom, a, b, becauseOfAtom, proofPart1)));
			prolog.addFact(prolog.compound(implies, b, c, prolog.list(impliesAtom, b, c, becauseOfAtom, proofPart2)));
			prolog.addFact(prolog.compound(implies, c, a, prolog.list(impliesAtom, c, a, becauseOfAtom, proofPart3)));

			// Rule
			prolog.addRule(prolog.compound(iff, A, B, prolog.list(iffAtom, A, B, becauseOfAtom, rule1, Proof1, Proof2)),
					prolog.conjunction(prolog.compound("implies", A, B, Proof1), prolog.compound("implies", B, A, Proof2)));

			prolog.addRule(prolog.compound(implies, A, C, prolog.list(impliesAtom, A, C, becauseOfAtom, transitiveAtom, Proof1, Proof2)),
					prolog.conjunction(prolog.compound("implies", A, B, Proof1), prolog.compound("implies", B, C, Proof2)));

			// implies(A,C, [implies,A,C, becauseof, transitiv, Proof1, Proof2]):- implies(A,B, Proof1), implies( B,C, Proof2).

			prolog.initEnd();
			System.out.println(prolog.getAddedObjectsAsRules());

			Object toBeProven = prolog.conjunction(//
					prolog.compound(implies, a, c, Proof));
			Map variables = prolog.getFirstSolution(toBeProven, Proof);
			Object theProof = variables.get(proofVar);
			Assert.assertNotNull(theProof);

			toBeProven = prolog.conjunction(//
					prolog.compound(iff, a, c, Proof));

			variables = prolog.getFirstSolution(toBeProven, Proof);
			theProof = variables.get(proofVar);
			Assert.assertNotNull(theProof);
			StringBuilder text = new StringBuilder();
			prolog.humanReadable(theProof, text);
			String string = text.toString();
			Assert.assertTrue(string.contains(ruleString));
			Assert.assertTrue(string.contains(proofPart1String));
			Assert.assertTrue(string.contains(proofPart2String));

		}


	}


}
