package net.sf.javaprinciples.graph.json;

import java.util.ArrayList;
import java.util.List;

import net.sf.javaprinciples.graph.GraphWriter;

/**
 * Writes a graph using the Javascript Object Notation
 *
 * @author Warwick Slade
 */
public class JsonGraphWriter implements GraphWriter
{
    private StringBuilder buffer = new StringBuilder();
    private boolean started = false;
    private List<String> errors = new ArrayList<String>();

    @Override
    public void writeString(String name, String value)
    {
        if (started)
        {
            buffer.append(",");
        }
        else
        {
            started = true;
        }

        writeString(name);
        writeSep();
        if (value == null)
        {
            buffer.append("null");
        }
        else
        {
            writeString(value);
        }
    }

    @Override
    public void writeArray(String name)
    {
        if (started)
        {
            buffer.append(",");
        }
        writeString(name);
        writeSep();
        buffer.append('[');
        started = false;
    }

    @Override
    public void writeArray()
    {
        buffer.append(']');
        started = true;
    }

    @Override
    public void writeObject(String name)
    {
        if (started)
        {
            buffer.append(",");
        }

        started = false;

        if (name.length() > 0)
        {
            writeString(name);
            writeSep();
        }
        buffer.append("{");
    }

    @Override
    public void writeObject()
    {
        buffer.append("}");
        started = true;
    }

    @Override
    public void reportError(String message)
    {
        errors.add(message);
    }

    @Override
    public boolean hasErrors()
    {
        return !errors.isEmpty();
    }

    public String result()
    {
        String ret = this.buffer.toString();

        this.buffer.setLength(0);

        return ret;
    }

    public String toString()
    {
        return buffer.toString();
    }

    protected void writeSep()
    {
        buffer.append(":");
    }

    protected void writeString(String value)
    {
        buffer.append('"');
        buffer.append(value);
        buffer.append('"');
    }
}
