package net.sf.javaprinciples.presentation.component;

import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.SpanElement;
import com.google.gwt.event.shared.EventHandler;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.web.bindery.event.shared.EventBus;

public class HeaderComponent
{
    private EventBus eventBus;
    private String userFirstName;
    private String userLastName;

    public HeaderComponent(EventBus eventBus)
    {
        this.eventBus = eventBus;

        eventBus.addHandler(StatusChangeEvent.TYPE, new StatusHandler()
        {
            @Override
            public void onStatusChange(StatusChangeEvent event)
            {
                manageStatus(event.getStatusMessage(), event.isShowStatus(), event.isError());
            }
        });
        eventBus.addHandler(LoginEvent.TYPE, new LoginHandler()
        {
            @Override
            public void onLogin(LoginEvent event)
            {
                userFirstName = event.getUserFirstName();
                userLastName = event.getUserLastName();
                ensureUsername();
            }
        });
    }

    private void manageStatus(String statusMessage, boolean showStatus, boolean error)
    {
        RootPanel rootPanel = RootPanel.get("status");
        rootPanel.clear();
        SpanElement span = Document.get().createSpanElement();
        span.setInnerText(statusMessage != null ? statusMessage : "");
        rootPanel.getElement().appendChild(span);
        manageStatusElement(rootPanel, showStatus, error);
    }

    private void manageStatusElement(RootPanel rootPanel, boolean showStatus, boolean error)
    {
        Element rootElement = rootPanel.getElement().getParentElement();
        if (rootElement != null)
        {
            showStatusIfNeeded(rootElement, showStatus);
            modifyAppearanceAccordingToStatus(rootPanel.getElement(), error);
        }
    }

    private void modifyAppearanceAccordingToStatus(Element statusElement, boolean error)
    {
        if (error)
        {
            statusElement.removeClassName("alert-success");
            statusElement.addClassName("alert-danger");
        }
        else
        {
            statusElement.removeClassName("alert-danger");
            statusElement.addClassName("alert-success");
        }
    }

    private void showStatusIfNeeded(com.google.gwt.dom.client.Element statusElement, boolean showStatus)
    {
        String style = statusElement.getClassName();
        if (showStatus && style != null && style.matches(".* ?hidden ?.*"))
        {
            statusElement.removeClassName("hidden");
        }
        if (!showStatus && style != null && !style.matches(".* ?hidden ?.*"))
        {
            statusElement.addClassName("hidden");
        }
    }

    private void ensureUsername()
    {
        RootPanel rootPanel = RootPanel.get("currentUser");
        if (rootPanel != null)
        {
            rootPanel.getElement().setInnerText("Welcome " + userFirstName + " " + userLastName);
        }
        Element elem = DOM.getElementById("authBar");
        if (elem != null)
        {
            elem.removeClassName("hidden");
        }
    }

    public interface StatusHandler extends EventHandler
    {
        void onStatusChange(StatusChangeEvent event);
    }

    public interface LoginHandler extends EventHandler
    {
        void onLogin(LoginEvent event);
    }

    public static class StatusChangeEvent extends GwtEvent<StatusHandler>
    {
        private String statusMessage;
        private boolean showStatus;
        private boolean error;

        public static Type<StatusHandler> TYPE = new Type<StatusHandler>();

        public StatusChangeEvent(String statusMessage, boolean showStatus, boolean error)
        {
            this.statusMessage = statusMessage;
            this.showStatus = showStatus;
            this.error = error;
        }

        @Override
        public Type<StatusHandler> getAssociatedType()
        {
            return TYPE;
        }

        @Override
        protected void dispatch(StatusHandler handler)
        {
            handler.onStatusChange(this);
        }

        public boolean isShowStatus()
        {
            return showStatus;
        }

        public String getStatusMessage()
        {
            return statusMessage;
        }

        public boolean isError()
        {
            return error;
        }
    }

    public static class LoginEvent extends GwtEvent<LoginHandler>
    {
        public static Type<LoginHandler> TYPE = new Type<LoginHandler>();

        private String userFirstName;
        private String userLastName;

        public LoginEvent(String userFirstName, String userLastName)
        {
            this.userFirstName = userFirstName;
            this.userLastName = userLastName;
        }

        @Override
        public Type<LoginHandler> getAssociatedType()
        {
            return TYPE;
        }

        @Override
        protected void dispatch(LoginHandler handler)
        {
            handler.onLogin(this);
        }

        public String getUserLastName()
        {
            return userLastName;
        }

        public String getUserFirstName()
        {
            return userFirstName;
        }
    }
}
