package net.sf.javaprinciples.presentation.control.model;

import net.sf.javaprinciples.graph.GraphReader;
import net.sf.javaprinciples.graph.GraphWriter;
import net.sf.javaprinciples.model.metadata.AttributeMetadata;
import net.sf.javaprinciples.model.shared.StringUtilsShared;
import net.sf.javaprinciples.presentation.activity.ClientContext;
import net.sf.javaprinciples.presentation.control.Control;
import net.sf.javaprinciples.presentation.control.ValueChangeListener;
import net.sf.javaprinciples.presentation.view.View;
import net.sf.javaprinciples.presentation.view.ViewFactory;

/**
 * A leaf control whose view will be determined by the metadata.
 *
 * @author Warwick Slade
 */
public class AttributeControl implements Control
{
    private AttributeMetadata attributeMetadata;
    private View parent;
    private View view;
    private ClientContext clientContext;
    private String content;

    private boolean visible;
    private boolean enabled;

    public AttributeControl(AttributeMetadata attributeMetadata, View parent, ClientContext clientContext)
    {
        this.attributeMetadata = attributeMetadata;
        this.parent = parent;
        this.clientContext = clientContext;
    }

    @Override
    public boolean isVisible()
    {
        return visible;
    }

    @Override
    public void setVisible(boolean visible)
    {
        this.visible = visible;
    }

    @Override
    public boolean isEnabled()
    {
        return enabled;
    }

    @Override
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
        view.setEnabled(enabled);
    }

    @Override
    public void bind(GraphReader reader)
    {
        content = reader.readString(attributeMetadata.getName());
        ViewFactory viewFactory = clientContext.getViewFactory();
        view = viewFactory.createView(attributeMetadata, content);

        visible = view.getWidget().isVisible();

        view.addValueChangedListener(new ValueChangeListener()
        {
            @Override
            public void valueChanged(String value)
            {
                content = value;
            }
        });
        parent.add(view);
    }

    @Override
    public void unbind(GraphWriter writer)
    {
        if (visible && !attributeMetadata.isOptional())
        {
            if (StringUtilsShared.isBlank(content))
            {
                writer.reportError(attributeMetadata.getLabel());
                // ToDo Localise
                view.reportError("This field requires a value!");
            }
            else
            {
                view.clearError();
            }
        }
        writer.writeString(attributeMetadata.getName(), content);
    }
}
