package net.sf.javaprinciples.presentation.control.model;

import java.util.ArrayList;
import java.util.List;
import net.sf.javaprinciples.graph.GraphReader;
import net.sf.javaprinciples.graph.GraphWriter;
import net.sf.javaprinciples.model.metadata.AttributeMetadata;
import net.sf.javaprinciples.model.metadata.ClassMetadata;
import net.sf.javaprinciples.model.metadata.ClassMetadataExtension;
import net.sf.javaprinciples.presentation.activity.ClientContext;
import net.sf.javaprinciples.presentation.control.Control;
import net.sf.javaprinciples.presentation.control.ControlContainer;
import net.sf.javaprinciples.presentation.view.View;
import net.sf.javaprinciples.presentation.view.model.CompositeView;
import net.sf.javaprinciples.presentation.view.model.MultipleInstanceView;

/**
 * A control whose model is based on an attribute with a 0..* multiplicity.
 *
 * @author Warwick Slade
 */
public class ClassControl implements ControlContainer
{
    ClientContext clientContext;
    AttributeMetadata attributeMetadata;
    List<AttributeMetadata> classAttributeMetadatas;
    boolean repeating;
    View parent;

    // Per instance
    List<Instance> instances = new ArrayList<Instance>();
    private boolean enabled;

    private enum Mixture
    {
        onlyAttributes,
        onlyClass,
        mixture
    };


    private class Instance
    {
        List<Control> controls = new ArrayList<Control>();
        View instanceView;

        public Instance(List<AttributeMetadata> classAttributeMetadatas)
        {
            // If there is a mixture of simple and complex types then we need to put all the simple
            // types first in their own view
            View attributeView = null;
            Mixture mixture = hasMixture(classAttributeMetadatas);
            if (mixture == Mixture.mixture)
            {
                instanceView = new CompositeView(clientContext, attributeMetadata);

                attributeView = new CompositeView("propertySheetClass");
                instanceView.add(attributeView);
            }
            else
            {
                if (mixture == Mixture.onlyClass)
                {
                    instanceView = new CompositeView(clientContext, attributeMetadata);
                }
                else
                {
                    instanceView = new CompositeView(clientContext, attributeMetadata, "propertySheetClass");
                }
                attributeView = instanceView;
            }

            for (AttributeMetadata classAttributeMetadata : classAttributeMetadatas)
            {
                ClassMetadataExtension classMetadataExtension = clientContext.getModelSupport().findExtension(classAttributeMetadata.getExtensions(), ClassMetadataExtension.class);

                if (classMetadataExtension != null)
                {
                    if (classAttributeMetadata.isHidden())
                    {
                        // This attribute is hidden so we wont display the associated class
                        continue;
                    }

                    ClassControl control = new ClassControl(classAttributeMetadata, instanceView, clientContext);
                    controls.add(control);
                }
                else
                {
                    AttributeControl control = new AttributeControl(classAttributeMetadata, attributeView, clientContext);
                    controls.add(control);
                }
            }
        }

        private Mixture hasMixture(List<AttributeMetadata> classAttributeMetadatas)
        {
            boolean oneAttribute = false;
            boolean oneClass = false;
            for (AttributeMetadata classAttributeMetadata : classAttributeMetadatas)
            {
                if (classAttributeMetadata.isHidden())
                {
                    // This attribute is hidden so we wont display the associated class
                    continue;
                }

                ClassMetadataExtension classMetadataExtension = clientContext.getModelSupport().findExtension(classAttributeMetadata.getExtensions(), ClassMetadataExtension.class);
                if (classMetadataExtension != null)
                {
                    if (oneAttribute)
                    {
                        return Mixture.mixture;
                    }
                    else
                    {
                        oneClass = true;
                    }
                }
                else
                {
                    if (oneClass)
                    {
                        return Mixture.mixture;
                    }
                    else
                    {
                        oneAttribute = true;
                    }
                }
            }
            return oneClass ? Mixture.onlyClass : Mixture.onlyAttributes;
        }

        public void bind(GraphReader reader)
        {
            for (Control control : controls)
            {
                control.bind(reader);
            }
        }

        public void unbind(GraphWriter writer)
        {
            for (Control control : controls)
            {
                control.unbind(writer);
            }
        }

        public void setEnabled(boolean enabled)
        {
            for (Control control : controls)
            {
                control.setEnabled(enabled);
            }
        }
    }


    public ClassControl(AttributeMetadata attributeMetadata, View parent, ClientContext clientContext)
    {
        this.clientContext = clientContext;
        this.attributeMetadata = attributeMetadata;
        this.parent = parent;

        ClassMetadataExtension extension = clientContext.getModelSupport().findExtension(attributeMetadata.getExtensions(), ClassMetadataExtension.class);
        ClassMetadata classMetadata = extension.getClazz();

        classAttributeMetadatas = classMetadata.getAttributes();

        int min = extension.getMultiplicityMinimum();
        int max = extension.getMultiplicityMaximum();
        repeating = max > 1;

        if (repeating)
        {
            for (int i = 0; i < min; i++)
            {
                Instance instance = new Instance(classAttributeMetadatas);
                instances.add(instance);
            }
        }
        else
        {
            Instance instance = new Instance(classAttributeMetadatas);
            instances.add(instance);
        }
    }

    @Override
    public Control findControlByName(String name)
    {
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    @Override
    public List<Control> getControls()
    {
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    @Override
    public boolean isVisible()
    {
        return false;  //To change body of implemented methods use File | Settings | File Templates.
    }

    @Override
    public void setVisible(boolean visible)
    {

    }

    @Override
    public boolean isEnabled()
    {
        return enabled;
    }

    @Override
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
        for (Instance instance : instances)
        {
            instance.setEnabled(enabled);
        }
    }

    @Override
    public void bind(GraphReader reader)
    {
        // Connect the views
        if (repeating)
        {
            MultipleInstanceView view = new MultipleInstanceView(attributeMetadata, repeating);
            parent.add(view);

            for (Instance instance : instances)
            {
                view.add(instance.instanceView);
            }
        }
        else
        {
            parent.add(instances.get(0).instanceView);
        }

        // Bind the data
        if (repeating)
        {
            reader.readArray(attributeMetadata.getName());
            for (Instance instance : instances)
            {
                instance.bind(reader);
                reader.readArray(null);
            }
            reader.readArray();
        }
        else
        {
            boolean hasResult = reader.readObject(attributeMetadata.getName());
            instances.get(0).bind(reader);
            if (hasResult)
            {
                reader.readObject();
            }
        }
    }

    @Override
    public void unbind(GraphWriter writer)
    {
        if (repeating)
        {
            writer.writeArray(attributeMetadata.getName());
            for (Instance instance : instances)
            {
                instance.unbind(writer);
                writer.writeArray(null);
            }
            writer.writeArray();
        }
        else
        {
            writer.writeObject(attributeMetadata.getName());
            instances.get(0).unbind(writer);
            writer.writeObject();
        }
    }
}
