package net.sf.javaprinciples.presentation.event;

import java.util.HashMap;
import java.util.Map;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.core.client.JsonUtils;
import com.google.gwt.event.shared.EventHandler;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.Response;
import com.google.web.bindery.event.shared.EventBus;

import net.sf.javaprinciples.model.metadata.AttributeMetadata;
import net.sf.javaprinciples.model.service.JsonModelServiceAsync;
import net.sf.javaprinciples.model.service.impl.JsonModelServiceAsyncImpl;
import net.sf.javaprinciples.presentation.activity.ClientContext;

/**
 * Encapsulates the retrieving of the model from the server.
 *
 * @author Warwick Slade
 */
public class ModelAdapter extends AbstractAdapter
{
    private final JsonModelServiceAsync modelService;
    private final String resourcePath;

    private Map<String, AttributeMetadata> models = new HashMap<String, AttributeMetadata>();

    public ModelAdapter(EventBus eventBus, String resourcePath)
    {
        super(eventBus);
        this.resourcePath = resourcePath;

        modelService = new JsonModelServiceAsyncImpl(resourcePath);

        eventBus.addHandler(ModelEvent.TYPE, new ModelEventHandler()
        {
            @Override
            public void onModel(ModelEvent modelEvent)
            {
                if (modelEvent.message != null)
                {
                    modelEvent.callback.modelFail(modelEvent.name, modelEvent.message);
                } else
                {
                    modelEvent.callback.modelReady(modelEvent.name, modelEvent.model);
                }
            }
        });
    }

    public interface ModelEventHandler extends EventHandler
    {
        void onModel(ModelEvent event);
    }

    public static class ModelEvent extends GwtEvent<ModelEventHandler>
    {
        private String name;
        private AttributeMetadata model;
        private ClientContext.ModelAsynchCall callback;
        private String message;

        public static Type<ModelEventHandler> TYPE = new Type<ModelEventHandler>();

        public ModelEvent(String name, AttributeMetadata model, ClientContext.ModelAsynchCall callback)
        {
            this.name = name;
            this.model = model;
            this.callback = callback;
        }

        public ModelEvent(String name, String message, ClientContext.ModelAsynchCall callback)
        {
            this.name = name;
            this.message = message;
            this.callback = callback;
        }

        public String getName()
        {
            return name;
        }

        public void setName(String name)
        {
            this.name = name;
        }

        public AttributeMetadata getModel()
        {
            return model;
        }

        public void setModel(AttributeMetadata model)
        {
            this.model = model;
        }

        @Override
        public Type<ModelEventHandler> getAssociatedType()
        {
            return TYPE;
        }

        @Override
        protected void dispatch(ModelEventHandler handler)
        {
            handler.onModel(this);
        }
    }

    //@Override
    public void loadModel(final String name, final ClientContext.ModelAsynchCall callback)
    {
        final AttributeMetadata model = models.get(name);
        if (model != null)
        {
            eventBus.fireEvent(new ModelEvent(name, model, callback));
            return;
        }

        try
        {
            modelService.findModel(name, new RequestCallback()
            {
                // This method presumes a certain format in the JSON response.
                private native AttributeMetadata parseResponse(JavaScriptObject result)
                  /*-{
                      return result;
                  }-*/;

                @Override
                public void onResponseReceived(Request request, Response response)
                {
                    if (200 == response.getStatusCode() || 201 == response.getStatusCode())
                    {
                        String responseText = response.getText();
                        if (!redirectHack(responseText))
                        {
                            AttributeMetadata result = parseResponse(JsonUtils.safeEval(responseText));
                            models.put(name, result);
                            eventBus.fireEvent(new ModelEvent(name, result, callback));
                        }
                    }
                    else if (Response.SC_UNAUTHORIZED == response.getStatusCode())
                    {
                        handleUnauthorised(response);
                    }
                    else
                    {
                        handleUnknown(response);
                    }
                }

                @Override
                public void onError(Request request, Throwable exception)
                {
                    eventBus.fireEvent(new ModelEvent(name, exception.getMessage(), callback));
                }
            });
        }
        catch (RuntimeException re)
        {
            eventBus.fireEvent(new ModelEvent(name, re.getMessage(), callback));
        }
    }

}