package net.sf.javaprinciples.presentation.event;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.EventHandler;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.RequestException;
import com.google.gwt.http.client.Response;
import com.google.web.bindery.event.shared.EventBus;

import net.sf.javaprinciples.presentation.activity.ClientContext;

/**
 * Encapsulates the retrieving of content from the server.
 *
 * @author Warwick Slade
 */
public class RetrieveContentAdapter extends AbstractAdapter
{
    String resourcePath;

    public RetrieveContentAdapter(EventBus eventBus, String resourcePath)
    {
        super(eventBus);
        this.resourcePath = resourcePath;

        eventBus.addHandler(RetrieveContentEvent.TYPE, new RetrieveContentEventHandler()
        {
            @Override
            public void onContent(RetrieveContentEvent contentEvent)
            {
                if (contentEvent.message != null)
                {
                    contentEvent.callback.contentFail(contentEvent.getIdentifier(), contentEvent.getMessage());
                }
                else
                {
                    contentEvent.callback.contentReady(contentEvent.getIdentifier(), contentEvent.getContent());
                }
            }
        });

    }

    public interface RetrieveContentEventHandler extends EventHandler
    {
        void onContent(RetrieveContentEvent event);
    }

    public static class RetrieveContentEvent extends GwtEvent<RetrieveContentEventHandler>
    {
        private String identifier;
        private String content;
        private ClientContext.RetrieveContentAsynchCall callback;
        private String message;

        public static Type<RetrieveContentEventHandler> TYPE = new Type<RetrieveContentEventHandler>();

        public RetrieveContentEvent(String identifier, String content, ClientContext.RetrieveContentAsynchCall callback)
        {
            this.identifier = identifier;
            this.content = content;
            this.callback = callback;
        }

        public RetrieveContentEvent(String identifier, String message, ClientContext.RetrieveContentAsynchCall callback, boolean failed)
        {
            this.identifier = identifier;
            this.message = message;
            this.callback = callback;
        }

        public String getIdentifier()
        {
            return identifier;
        }

        public void setIdentifier(String name)
        {
            this.identifier = name;
        }

        public String getContent()
        {
            return content;
        }

        public void setContent(String content)
        {
            this.content = content;
        }

        @Override
        public Type<RetrieveContentEventHandler> getAssociatedType()
        {
            return TYPE;
        }

        @Override
        protected void dispatch(RetrieveContentEventHandler handler)
        {
            handler.onContent(this);
        }

        public String getMessage()
        {
            return message;
        }
    }

    public void retrieveContent(final String identifier, final String query, final ClientContext.RetrieveContentAsynchCall callback)
    {
        String url = GWT.getModuleBaseURL() + resourcePath + "/query/" + identifier;

        // Send request to server and catch any errors.
        RequestBuilder builder = new RequestBuilder(RequestBuilder.POST, url);

        builder.setHeader("Content-type", "application/x-www-form-urlencoded");
        try
        {
            Request request = builder.sendRequest(query, new RequestCallback()
            {
                public void onError(Request request, Throwable exception)
                {
                    eventBus.fireEvent(new RetrieveContentEvent(identifier, exception.getMessage(), callback, true));
                }

                public void onResponseReceived(Request request, Response response)
                {
                    if (200 == response.getStatusCode() || 201 == response.getStatusCode())
                    {
                        String content = response.getText();
                        if (!redirectHack(content))
                        {
                            eventBus.fireEvent(new RetrieveContentEvent(identifier, content, callback));
                        }
                    }
                    else if (Response.SC_UNAUTHORIZED == response.getStatusCode())
                    {
                        handleUnauthorised(response);
                    }
                    else
                    {
                        handleUnknown(response);
                    }
                }
            });
        }
        catch (RequestException exception)
        {
            eventBus.fireEvent(new RetrieveContentEvent(identifier, exception.getMessage(), callback, true));
        }
    }

}
