package net.sf.javaprinciples.presentation.event;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.EventHandler;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.RequestException;
import com.google.gwt.http.client.Response;
import com.google.web.bindery.event.shared.EventBus;

import net.sf.javaprinciples.presentation.activity.ClientContext;

/**
 * Encapsulates the storing of content to the server
 *
 * @author Warwick Slade
 */
public class StoreContentAdapter extends AbstractAdapter
{
    String resourcePath;

    public StoreContentAdapter(EventBus eventBus, String resourcePath)
    {
        super(eventBus);

        this.resourcePath = resourcePath;

        eventBus.addHandler(StoreContentAdapter.StoreContentEvent.TYPE, new StoreContentAdapter.StoreContentEventHandler()
        {
            @Override
            public void onContent(StoreContentAdapter.StoreContentEvent contentEvent)
            {
                if (contentEvent.failed)
                {
                    contentEvent.callback.storeFail(contentEvent.getIdentifier(), contentEvent.getMessage());
                }
                else
                {
                    contentEvent.callback.storeSuccess(contentEvent.getIdentifier(), contentEvent.getMessage());
                }
            }
        });

    }

    public void storeContent(final String identifier, String content, final ClientContext.StoreContentAsynchCall callback)
    {
        String url = GWT.getModuleBaseURL() + resourcePath + "/content/" + identifier;

        // Send request to server and catch any errors.
        RequestBuilder builder = new RequestBuilder(RequestBuilder.POST, url);

        builder.setHeader("Content-type", "application/x-www-form-urlencoded");
        try
        {
            Request request = builder.sendRequest(content, new RequestCallback()
            {
                public void onError(Request request, Throwable exception)
                {
                    eventBus.fireEvent(new StoreContentAdapter.StoreContentEvent(identifier, exception.getMessage(), callback, true));
                }

                public void onResponseReceived(Request request, Response response)
                {
                    if (200 == response.getStatusCode() || 201 == response.getStatusCode())
                    {
                        String content = response.getText();
                        if (!redirectHack(content))
                        {
                            eventBus.fireEvent(new StoreContentAdapter.StoreContentEvent(identifier, content, callback, false));
                        }
                    }
                    else if (Response.SC_UNAUTHORIZED == response.getStatusCode())
                    {
                        handleUnauthorised(response);
                    }
                    else if (203 == response.getStatusCode())
                    {
                        String content = response.getHeader("X-Application-Error-Code");
                        eventBus.fireEvent(new StoreContentAdapter.StoreContentEvent(identifier, content, callback, true));
                    }
                    else
                    {
                        eventBus.fireEvent(new StoreContentAdapter.StoreContentEvent(identifier, response.getStatusText(), callback, true));
                    }
                }
            });
        }
        catch (RequestException exception)
        {
            eventBus.fireEvent(new StoreContentAdapter.StoreContentEvent(identifier, exception.getMessage(), callback, true));
        }
    }

    public interface StoreContentEventHandler extends EventHandler
    {
        void onContent(StoreContentEvent event);
    }

    public static class StoreContentEvent extends GwtEvent<StoreContentEventHandler>
    {
        private String identifier;
        private ClientContext.StoreContentAsynchCall callback;
        private String message;
        private boolean failed;

        public static Type<StoreContentEventHandler> TYPE = new Type<StoreContentEventHandler>();

        public StoreContentEvent(String identifier, ClientContext.StoreContentAsynchCall callback)
        {
            this.identifier = identifier;
            this.callback = callback;
        }

        public StoreContentEvent(String identifier, String message, ClientContext.StoreContentAsynchCall callback, boolean failed)
        {
            this.identifier = identifier;
            this.message = message;
            this.callback = callback;
            this.failed = failed;
        }

        public String getIdentifier()
        {
            return identifier;
        }

        public void setIdentifier(String name)
        {
            this.identifier = name;
        }

        @Override
        public Type<StoreContentEventHandler> getAssociatedType()
        {
            return TYPE;
        }

        @Override
        protected void dispatch(StoreContentEventHandler handler)
        {
            handler.onContent(this);
        }

        public String getMessage()
        {
            return message;
        }
    }
}
