package net.sf.javaprinciples.presentation.view.model;

import com.google.gwt.user.client.ui.ComplexPanel;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.FocusWidget;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;

import net.sf.javaprinciples.model.metadata.AttributeMetadata;
import net.sf.javaprinciples.model.shared.StringUtilsShared;
import net.sf.javaprinciples.presentation.control.ValueChangeListener;
import net.sf.javaprinciples.presentation.view.View;
import net.sf.javaprinciples.presentation.widget.WidgetFactory;

/**
 * Modify the parts of a standard view factory so that the entry to the field
 * is confirmed by the user entering the data twice.
 *
 * @author Warwick Slade
 */
public class ConfirmationViewFactory extends PropertyViewFactory
{

    public static final String CONFIRMATION_PREFIX = "confirm-";

    public ConfirmationViewFactory(WidgetFactory widgetFactory)
    {
        super(widgetFactory);
    }

    private class CompositeView extends PropertyView
    {
        String label;
        String value1;
        String value2;
        PropertyView view1 = new PropertyView();
        PropertyView view2 = new PropertyView();

        public CompositeView(String label)
        {
            this.label = label;
            view1.addValueChangedListener(new ValueChangeListener()
            {
                @Override
                public void valueChanged(String value)
                {
                    value1 = value;
                    detectDone();
                }
            });
            view2.addValueChangedListener(new ValueChangeListener()
            {
                @Override
                public void valueChanged(String value)
                {
                    value2 = value;
                    detectDone();
                }
            });
        }

        protected void detectDone()
        {
            if (value1 == null || value2 == null)
            {
                valueChanged(null);
            }
            else if (value1.equals(value2))
            {
                valueChanged(value1);
            }
            else
            {
                valueChanged(null);
            }
        }

        public void reportError(String message)
        {
            if (!StringUtilsShared.isBlank(value1) && !StringUtilsShared.isBlank(value2))
            {
                message = "The confirmation " + label +" does not match";
            }
            super.reportError(message);
        }

        @Override
        public void setEnabled(boolean enabled)
        {
            ((FocusWidget)view1.getInput()).setEnabled(enabled);
            ((FocusWidget)view2.getInput()).setEnabled(enabled);
        }
    }

    @Override
    public View createView(AttributeMetadata attributeMetadata, String content)
    {
        CompositeView view = new CompositeView(attributeMetadata.getLabel());

        // 1. Create a holder for both inputs
        ComplexPanel container = makeContainer();
        view.setContainer(container);

        Label errorLabel = makeErrorLabel(attributeMetadata);
        view.setErrorMessage(errorLabel);
        container.add(errorLabel);

        // 1.5. Create holder for first input
        ComplexPanel subcontainer = new FlowPanel();
        container.add(subcontainer);

        // 2. Create a label
        Widget label = makeLabel(attributeMetadata);
        view.view1.setLabel(label);
        subcontainer.add(label);

        // 3. Create an input
        Widget input = makeInput(attributeMetadata, view.view1, content);
        view.view1.setInput(input);
        subcontainer.add(input);

        // 3.5. Create holder for second input
        subcontainer = new FlowPanel();
        container.add(subcontainer);

        // 4. Create the confirmation label
        Widget label2 = makeConfirmationLabel(attributeMetadata);
        view.view2.setLabel(label);
        subcontainer.add(label2);

        // 5. Create the confirmation input
        Widget input2 = makeInput(attributeMetadata, view.view2, content, CONFIRMATION_PREFIX);
        view.view2.setInput(input);
        subcontainer.add(input2);

        // 4. Create a tooltip

        if (attributeMetadata.isHidden())
        {
            container.setVisible(false);
        }

        return view;
    }

    protected Widget makeConfirmationLabel(AttributeMetadata attributeMetadata)
    {
        return makeLabel("Confirm " + attributeMetadata.getLabel(), CONFIRMATION_PREFIX + attributeMetadata.getName());
    }
}
