package net.sf.javaprinciples.presentation.widget;

import java.util.Date;

import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HasEnabled;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.Widget;

import net.sf.javaprinciples.model.metadata.AttributeMetadata;
import net.sf.javaprinciples.model.shared.StringUtilsShared;
import net.sf.javaprinciples.presentation.control.ValueChangeListener;

/**
 * Create a widget based on a dd / mm / yyyy
 *
 * @author Warwick Slade
 */
public class DateOfBirthWidgetFactory implements WidgetFactory
{
    private static final DateTimeFormat presentationFormat = DateTimeFormat.getFormat(StringUtilsShared.DATE_FORMAT_PRESENTATION);
    private static final DateTimeFormat internalFormat = DateTimeFormat.getFormat(StringUtilsShared.DATE_FORMAT_INTERNAL);

    @Override
    public Widget createWidget(AttributeMetadata metadata,
                               final ValueChangeListener listener,
                               String content)
    {
        if (metadata.isReadOnly())
        {
            if (!StringUtilsShared.isBlank(content))
            {
                content = presentationFormat.format(internalFormat.parse(content));
            }
            return new Label(content != null ? content : "");
        }

        DateOfBirthWidget dateOfBirthWidget = new DateOfBirthWidget(metadata);
        dateOfBirthWidget.setValue(content);
        dateOfBirthWidget.addValueChangeHandler(new ValueChangeHandler<String>()
        {
            @Override
            public void onValueChange(ValueChangeEvent<String> event)
            {
                listener.valueChanged(event.getValue());
            }
        });

        return dateOfBirthWidget;
    }

    public static class DateOfBirthWidget extends FlowPanel implements HasValueChangeHandlers<String>, HasEnabled
    {
        private ListBox dayOfMonth;
        private ListBox month;
        private ListBox year;
        private boolean enabled = true;
        private Date date;

        public DateOfBirthWidget(AttributeMetadata metadata)
        {
            setStyleName("propertySheetInputCustom");

            dayOfMonth = new ListBox();
            dayOfMonth.setStylePrimaryName("internalList");
            fillListBox(dayOfMonth);
            add(dayOfMonth);

            month = new ListBox();
            month.setStylePrimaryName("internalList");
            fillMonthListBox(month);
            add(month);

            year = new ListBox();
            year.setStylePrimaryName("internalList");
            fillYearListBox(year);
            add(year);

            ChangeHandler handler = new ChangeHandler()
            {
                @Override
                public void onChange(ChangeEvent event)
                {
                    String sdayOfMonthValue = dayOfMonth.getValue(dayOfMonth.getSelectedIndex());
                    String smonth = month.getValue(month.getSelectedIndex());
                    String syear = year.getValue(year.getSelectedIndex());

                    if (sdayOfMonthValue.length() == 0 ||
                            smonth.length() == 0 ||
                            syear.length() == 0)
                    {
                        date = null;
                    }
                    else
                    {
                        int dayOfMonthValue = Integer.parseInt(sdayOfMonthValue);
                        int imonth = month.getSelectedIndex();
                        int year = Integer.parseInt(syear);
                        date = new Date(year - 1900, imonth -1 , dayOfMonthValue);
                    }
                    ValueChangeEvent.fire(DateOfBirthWidget.this, getValue());
                }

            };
            dayOfMonth.addChangeHandler(handler);
            month.addChangeHandler(handler);
            year.addChangeHandler(handler);
        }

        private void fillMonthListBox(ListBox box)
        {
            String[] months = LocaleInfo.getCurrentLocale().getDateTimeFormatInfo().monthsFullStandalone();
            box.addItem("month", "");
            for (String month : months)
            {
                box.addItem(month, month);
            }
        }

        private void fillYearListBox(ListBox box)
        {
            box.addItem("year", "");

            int now = new Date().getYear() + 1900;
            for (int year = now; year > now - 100; year--)
            {
                box.addItem(Integer.toString(year), Integer.toString(year));
            }
        }

        protected void fillListBox(ListBox box)
        {
            box.addItem("day", "");
            for (int i = 1; i <= 31; i++)
            {
                box.addItem(Integer.toString(i), Integer.toString(i));
            }
        }
        @Override
        public HandlerRegistration addValueChangeHandler(
                ValueChangeHandler<String> handler)
        {
            return this.addHandler(handler, ValueChangeEvent.getType());
        }

        public String getValue()
        {
            if (date == null)
            {
                return "";
            }
            String svalue = internalFormat.format(date);
            return svalue;
        }

        public void setValue(String identifier)
        {
            if (identifier == null || identifier.length() == 0)
            {
                return;
            }

            date = internalFormat.parse(identifier);

            int iday = Integer.parseInt(DateTimeFormat.getFormat( "d" ).format(date));

            dayOfMonth.setSelectedIndex(iday);
            month.setSelectedIndex(date.getMonth() + 1);

            int now = new Date().getYear();
            int iyear = now - date.getYear() + 1;
            year.setSelectedIndex(iyear);

            ValueChangeEvent.fire(DateOfBirthWidget.this, getValue());
        }

        public int getCost()
        {
            return 0;
        }

        @Override
        public boolean isEnabled()
        {
            return enabled;
        }

        @Override
        public void setEnabled(boolean enabled)
        {
            if (this.enabled == enabled)
            {
                return;
            }
            this.enabled = enabled;
            dayOfMonth.setEnabled(enabled);
            month.setEnabled(enabled);
            year.setEnabled(enabled);
        }
    }
}
