package net.sf.jcc.model.parser;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;

/**
 * This object is used for transferring information about the current parse
 * state between ElementHandler instances.
 * 
 * @author dcharlt
 * 
 */
public abstract class ParseContext
{

    private ParseContext root;
    private ParseContext parent;
    private ParsedElement parsedElement;
    private InternalStore<?> internalStore;
    private Deque<String> actionNames;

    private Map<String, Object> variables = new HashMap<String, Object>();

    /**
     * Constructs a new ParseContext.
     * 
     * @param parent The ParseContext of the element which is the parent to this
     *            element.
     * @param parsedElement the element that has been parsed.
     * @param patternFactory The pattern factory that we will use to locate
     *            patterns.
     */
    public ParseContext(ParseContext parent,
            ParsedElement parsedElement)
    {
        this.parent = parent;
        if (parent == null)
        {
            root = this;
        }
        else
        {
            root = parent.root;
        }

        this.parsedElement = parsedElement;
        this.actionNames = new ArrayDeque<String>();
    }

    /**
     * Returns the element that has been parsed as part of this parseContext.
     * 
     * @return Returns the element that has been parsed as part of this
     *         parseContext.
     */
    public ParsedElement getParsedElement()
    {
        return this.parsedElement;
    }

    public ParseContext getParent()
    {
        return parent;
    }

    /**
     * If the parent object is not null, set the internal store on the parent
     * object, otherwise set the internal store on this object.
     * 
     * @param internalStore The internal store to set
     */
    public void setInternalStore(InternalStore<?> internalStore)
    {
        if (root.internalStore != null)
        {
            throw new UnexpectedException(
                    "An InternalStore has already been configured for this ParseContext.");
        }

        root.internalStore = internalStore;

    }

    /**
     * If the parent object is not null, get the internal store from the parent
     * object, otherwise get the internal store from this object.
     * 
     * @return Returns the internal store.
     */
    public InternalStore<?> getInternalStore()
    {
        return root.internalStore;
    }

    /**
     * Adds a variable to the ultimate parent ParseContext from which this
     * ParseContext has descended.
     * 
     * @param name The name to use for the variable.
     * @param variable The value to use for the variable.
     */
    public void addGlobalVariable(String name, Object variable)
    {
        root.variables.put(name, variable);
    }

    /**
     * Adds a variable to the ParseContext.
     * 
     * @param name The name to use for the variable.
     * @param variable The value to use for the variable.
     */
    public void addVariable(String name, Object variable)
    {
        variables.put(name, variable);
    }

    /**
     * Returns true if the ParseContext contains a variable identified by name.
     * Initially looks to see if the variable is available within the
     * ParseContext, if not, it looks to see whether it is available from the
     * parent ParseContext recursively.
     * 
     * @param name The name of the variable.
     * @return Returns true if the variable was found, otherwise false.
     */
    public boolean containsVariable(String name)
    {
        if (variables.containsKey(name))
        {
            return true;
        }

        if (this != root && root.containsVariable(name))
        {
            return true;
        }

        if (parent != null)
        {
            return parent.containsVariable(name);
        }

        return false;
    }

    /**
     * Retrieves a variable by name. Only considers this context.
     * 
     * @param name The name of the variable to return.
     * @return Returns the value of the variable or null if it was not found.
     */
    public Object getLocalVariable(String name)
    {
    	return variables.get(name);
    }
    
    /**
     * Retrieves a variable by name. Initially looks to see if the variable is
     * available within the ParseContext, if not, it attempts to retrieve the
     * variable from the parents context.
     * 
     * @param name The name of the variable to return.
     * @return Returns the value of the variable or null if it was not found.
     */
    public Object getVariable(String name)
    {
        if (variables.containsKey(name))
        {
            return variables.get(name);
        }

        if (this != root && root.containsVariable(name))
        {
            return root.getVariable(name);
        }

        if (parent != null)
        {
            return parent.getVariable(name);
        }

        return null;
    }

    /**
     * Removes the most recent action name from the stack.
     * 
     * @return The most recent action name.
     */
    public String popActionName()
    {
        if (root.actionNames.isEmpty())
        {
            return null;
        }
        return root.actionNames.pop();
    }

    /**
     * Pushes the most recent action name to the stack.
     * 
     * @param currentActionName The name to push.
     */
    public void pushActionName(String currentActionName)
    {
        if (currentActionName == null)
        {
            return;
        }
        root.actionNames.push(currentActionName);
    }

    /**
     * Returns the stack containing all of the action names in current context.
     * 
     * @return The stack of names.
     */
    public Deque<String> getActionNames()
    {
        return root.actionNames;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode()
    {
        HashCodeBuilder builder = new HashCodeBuilder();
        builder.append(parent);
        builder.append(parsedElement);
        builder.append(internalStore);
        builder.append(variables);
        return builder.toHashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        ParseContext other = (ParseContext) obj;

        EqualsBuilder builder = new EqualsBuilder();
        builder.append(parent, other.parent);
        builder.append(parsedElement, other.parsedElement);
        builder.append(internalStore, other.internalStore);
        builder.append(variables, other.variables);

        return builder.isEquals();

    }
}
