package net.sf.jcc.model.parser;

import javax.xml.namespace.QName;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * Implementation of ParsedElement that wraps an XML StartElement.
 *
 * @author dcharlt
 *
 */
public class XMLParsedElement implements ParsedElement
{

    private final StartElement element;
    private final ElementName elementName;
    private final String elementText;

    /**
     * Construct a new XMLParsedElement.
     *
     * @param element the element that has been parsed.
     * @param nextEvent event that follows the parsed element.
     */
    public XMLParsedElement(StartElement element, XMLEvent nextEvent)
    {
        assert element != null;

        this.element = element;
        this.elementName = new ElementName(element.getName().getNamespaceURI(),
                element.getName().getLocalPart());
        this.elementText = getElementText(nextEvent);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ElementName getName()
    {
        return elementName;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getText()
    {
        return elementText;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getAttributeText(ElementName elementName)
    {
        String value = null;
        Attribute attribute = element.getAttributeByName(
                new QName(elementName.getNamespace(), elementName.getName()));

        QName name = element.getName();
        if (attribute == null && name != null && name.getNamespaceURI() != null
                && name.getNamespaceURI().equals(elementName.getNamespace()))
        {
            attribute = element.getAttributeByName(
                    new QName(null, elementName.getName()));

        }

        if (attribute != null)
        {
            value = attribute.getValue();
        }

        return value;
    }

    private String getElementText(XMLEvent nextEvent)
    {
        if (nextEvent != null && nextEvent.isCharacters()
                && !nextEvent.asCharacters().isWhiteSpace())
        {
            String data = nextEvent.asCharacters().getData();
            if (data != null)
            {
                data = data.trim();
            }
            return data;
        }

        return null;
    }

    public StartElement getElement()
    {
        return element;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString()
    {
        ToStringBuilder builder = new ToStringBuilder(this);
        builder.append(elementName);
        builder.append(elementText);
        return builder.toString();
    }

}
