package net.sf.jcc.model.parser;

import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

/**
 * A parser that takes XML data and uses a {@link TransformationRegistry} and
 * {@link PatternFactory} to create a new file containing the result of
 * transformation.
 *
 * @author omadgwi
 */
public class XmlParser
{
    private final TransformationRegistry transformationRegistry;
    private final XMLInputFactory xmlInputFactory;
    private final InternalStoreFactory internalStoreFactory;

    /**
     * @param transformationRegistry The {@code TransformationRegistry}.
     * @param patternFactory The {@code PatternFactory}.
     * @param xmlInputFactory The {@code XMLInputFactory}.
     * @param internalStoreFactory The {@code InternalStoreFactory}.
     */
    public XmlParser(TransformationRegistry transformationRegistry,
            XMLInputFactory xmlInputFactory,
            InternalStoreFactory internalStoreFactory)
    {
        this.transformationRegistry = transformationRegistry;
        this.xmlInputFactory = xmlInputFactory;
        this.internalStoreFactory = internalStoreFactory;
    }

    /**
     * Parse the XML data from the given {@code InputStream}, returning the file
     * containing the transformation result.
     *
     * @param inputStream The XML data to parse.
     * @param initialProperties Map of initial context variables to be added to the global variables collection.
     * @param transformResult The strategy to use for exporting the transform's result.
     */
    public void parse(InputStream inputStream, Map<String, Object> initialProperties,
            TransformationResult transformResult)
    {
        assert inputStream != null;

        try
        {
            transform(inputStream, initialProperties, transformResult);
        }
        catch (XMLStreamException e)
        {
            throw new UnexpectedException(e.getMessage(), e);
        }
    }

    private void transform(InputStream inputStream, Map<String, Object> initialProperties,
            TransformationResult transformResult) throws XMLStreamException
    {
        XMLParseContext parseContext = null;
        XMLEventReader eventReader = xmlInputFactory.createXMLEventReader(inputStream);
        try
        {
            Map<String, Exception> errorMap = new LinkedHashMap<String, Exception>();
            ElementHandler handler = obtainRootOfHandledNodes(eventReader);
            StartElement event = eventReader.nextEvent().asStartElement();
            parseContext = createParseContext(eventReader, event, initialProperties);
            handleElement(event, handler, parseContext, errorMap);
            handleChildren(event.getName(), parseContext, eventReader, false,
                    errorMap);
            obtainTransformationResult(parseContext, transformResult);
            ensureThereAreNoMoreElementsToHandle(eventReader);

            for (Map.Entry<String, Exception> errorEntry : errorMap
                    .entrySet())
            {
                System.err.println("Failed to handle event " + 
                        errorEntry.getValue() + " : " + errorEntry.getKey());
            }
            if (!errorMap.isEmpty())
            {
                throw new UnexpectedException(
                        "[" + errorMap.size() +
                            "] Errors occured while handling elements. Refer to the log file for details.");
            }
        }
        finally
        {
            close(eventReader);
            close(parseContext);
        }
    }

    private void handleElement(XMLEvent event, ElementHandler handler,
            XMLParseContext parseContext, Map<String, Exception> errorMap)
    {
        try
        {
            handler.handleElement(parseContext);
        }
        catch (ElementHandlerException e)
        {
            errorMap.put(getPathStringForParseContext(parseContext), e);
        }
        catch (RuntimeException e)
        {
            errorMap.put(getPathStringForParseContext(parseContext), e);
        }
    }

    private String getPathStringForParseContext(XMLParseContext parseContext)
    {
        StringBuilder sb = new StringBuilder();
        String seperator = "";
        for (ParseContext curContext = parseContext; curContext != null; curContext = curContext
                .getParent())
        {
            sb.insert(0, curContext.getParsedElement().getName() + seperator);
            seperator = ":";
        }
        String parseContextPath = sb.toString();
        return parseContextPath;
    }

    private void obtainTransformationResult(ParseContext parseContext, TransformationResult transformationResult)
    {
        InternalStore<?> internalStore = parseContext.getInternalStore();
        if (internalStore == null)
        {
            throw new UnexpectedException("Could not obtain Internal Store");
        }
        internalStore.export(transformationResult);
    }

    private ElementHandler obtainRootOfHandledNodes(XMLEventReader eventReader)
            throws XMLStreamException
    {
        ElementHandler handler = obtainNextElementHandler(eventReader);
        if (handler == null)
        {
            throw new UnexpectedException("No elements were handled.");
        }
        return handler;
    }

    private void ensureThereAreNoMoreElementsToHandle(XMLEventReader eventReader)
            throws XMLStreamException
    {
        if (obtainNextElementHandler(eventReader) != null)
        {
            throw new UnexpectedException("Multiple elements were handled.");
        }
    }

    /**
     * Iterate through the event reader searching for elements with a registered
     * handler. If a handler is found for an element it is returned and the
     * element event can be obtained by calling the
     * {@link XMLEventReader#nextEvent()} method.
     *
     * @param eventReader The reader to iterate with.
     * @return The next element handler to use.
     * @throws XMLStreamException If something goes wrong reading the XML.
     */
    private ElementHandler obtainNextElementHandler(XMLEventReader eventReader)
            throws XMLStreamException
    {
        while (eventReader.hasNext())
        {
            XMLEvent event = eventReader.peek();
            if (event.isStartElement())
            {
                ParseContext parseContext = createParseContext(eventReader,
                        event);
                ElementHandler handler = transformationRegistry
                        .lookupElementHandler(parseContext);
                if (handler != null)
                {
                    return handler;
                }
            }
            eventReader.nextEvent();
        }
        return null;
    }

    private void handleChildren(QName parentName,
            XMLParseContext parentContext,
            XMLEventReader eventReader, boolean skipElement,
            Map<String, Exception> errorMap)
            throws XMLStreamException
    {
        while (eventReader.hasNext())
        {
            XMLEvent event = eventReader.nextEvent();

            if (event.isStartElement())
            {
                XMLParseContext childParseContext = createParseContext(
                        parentContext, eventReader, event);

                // If we are skipping the element, do not lookup the
                // ElementHandler
                ElementHandler handler =
                        skipElement ? null : transformationRegistry
                        .lookupElementHandler(childParseContext);

                if (handler != null)
                {
                    childParseContext.pushActionName(handler.getName());
                    handleElement(event, handler, childParseContext, errorMap);
                    handleChildren(event.asStartElement().getName(),
                            childParseContext, eventReader, false, errorMap);
                    childParseContext.popActionName();
                }
                else
                {
                    if (!skipElement)
                    {
//                        Loggers.workflow
//                                .itil2(
//                                        "No ElementHandler found for event [{2}]. All child events will be ignored.",
//                                        "XP-HC-01", event.asStartElement()
//                                                .getName());
                    }
                    // Iterate through the children of this element but do not
                    // process any of them.
                    handleChildren(event.asStartElement().getName(),
                            childParseContext, eventReader, true, errorMap);
                }
            }
            else if (event.isEndElement())
            {
                QName qName = event.asEndElement().getName();
                if (qName.equals(parentName))
                {
                    break;
                }
            }
        }
    }

    private XMLParseContext createParseContext(XMLEventReader eventReader, StartElement event,
            Map<String, Object> initialProperties)
    {
        XMLParseContext context = createParseContext(eventReader, event);
        if (initialProperties != null)
        {
            for (Entry<String, Object> entry : initialProperties.entrySet())
            {
                context.addGlobalVariable(entry.getKey(), entry.getValue());
            }
        }
        context.setInternalStore(internalStoreFactory.createInternalStore());
        
        return context;
    }

    private XMLParseContext createParseContext(XMLEventReader eventReader,
            XMLEvent event)
    {
        return createParseContext(null, eventReader, event);
    }

    private XMLParseContext createParseContext(ParseContext parentContext,
            XMLEventReader eventReader, XMLEvent event)
    {
        XMLParseContext context = new XMLParseContext(parentContext, event.asStartElement(),
                eventReader);
        
        return context;
    }

    private void close(XMLEventReader eventReader)
    {
        try
        {
            eventReader.close();
        }
        catch (XMLStreamException e)
        {
//            Loggers.connectivity.error("Error closing XMLEventReader.", e,
//                    "XP-C-01");
        }
    }

    private void close(ParseContext parseContext)
    {
        if (parseContext != null && parseContext.getInternalStore() != null)
        {
            parseContext.getInternalStore().close();
        }
    }
}
