package net.sf.jcc.model.parser.uml2;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.StartElement;

import org.apache.commons.lang.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.xml.namespace.QNameEditor;

import net.sf.jcc.model.parser.ElementHandler;
import net.sf.jcc.model.parser.ElementHandlerException;
import net.sf.jcc.model.parser.ParseContext;
import net.sf.jcc.model.parser.XMLParsedElement;

public class ExtensionHandler implements ElementHandler
{
	public static final String PARENT = "ext.parent";
	private static final Logger log = LoggerFactory.getLogger(ExtensionHandler.class);
	private Map<String, Class<?>> elementMap;
	
	@Override
	public String getName() 
	{
		return getClass().getSimpleName();
	}
	
	@Override
	public void handleElement(ParseContext parseContext)
			throws ElementHandlerException 
	{
		XMLParsedElement pElement = (XMLParsedElement)parseContext.getParsedElement();
		
		StartElement element = pElement.getElement();

		boolean containedInParent = false;
		Class<?> umlClass = null;
		ParseContext parentContext = parseContext.getParent();
		Object parent = null;
		if (parentContext != null)
		{
			parent = parentContext.getLocalVariable(PARENT);
			if (parent != null)
			{
				QName thisElementName = element.getName();
				String classname = findClassOfAttribute(parent, thisElementName.getLocalPart());
				if (classname != null)
				{
					try
					{
						umlClass = (Class<?>) Class.forName(classname);
						containedInParent = true;
					}
					catch (ClassNotFoundException e) 
					{
						log.error("Class not found:" + classname);
						return;
					} 
				}
			}
		}
		if (umlClass == null)
		{
			umlClass = elementMap.get(element.getName().toString());
			if (umlClass == null)
			{
				log.debug("Class mapping not found:" + element.getName().toString());
				return;
			}
		}
		
		try 
		{
			Object e = umlClass.newInstance();
			
			assignProperties(e, element);
			if (containedInParent)
			{
				assignToParent(parent, element.getName().getLocalPart(), e);
			}
			else if (parentContext != null)
			{
				// Check if this a collection element Eg <tags><tag><tag></tags>
				QName thisElementName = element.getName();
				String parentName = parentContext.getParsedElement().getName().getName();
				if (parentName.startsWith(thisElementName.getLocalPart()))
				{
					parent = parentContext.getVariable(PARENT);
					if (parent != null)
					{
						assignToParent(parent, parentName, e);
					}
				}
			}
			parseContext.addVariable(PARENT, e);
			
			// Wire up to original.
			ElementStore store =  (ElementStore) parseContext.getInternalStore();
			Attribute attr = element.getAttributeByName(new QName("http://schema.omg.org/spec/XMI/2.1", "idref"));
			if (attr != null)
			{	
				ModelElement original = store.get(attr.getValue());
				if (original != null)
				{
					original.setExtension(e);
				}
				else
				{
                    // Some EA extensions do not relate to a piece of UML
                    // For example the notes on a diagram
                    // For the moment ignore as we only ever want UML model elements.
				}
			}
		} 
		catch (InstantiationException e) 
		{
			log.error("Class not instantiated:" + umlClass.getName());
		} 
		catch (IllegalAccessException e) 
		{
			log.error("Illegal Access:" + umlClass.getName());
		}
	}

	private void assignToParent(Object target, String name, Object value) 
	{
		BeanWrapper wrapper = new BeanWrapperImpl(target);
		try
		{
			Object descendant = wrapper.getPropertyValue(name);
			
			if (descendant instanceof List<?>)
			{
				List<Object> list = (List<Object>)descendant;
				list.add(value);
				return;
			}
		}
		catch (Exception ex)
		{
			// ok
		}
		try
		{
			wrapper.setPropertyValue(name, value);
		}
		catch (Exception ex)
		{
			log.info("Error writing property value:" + name + " " + target.getClass().getName());
		}
	}

	protected String findClassOfAttribute(Object target, String name) 
	{
		PropertyDescriptor pd = BeanUtils.getPropertyDescriptor(target.getClass(), name);
		if (pd == null)
		{
			return null;
		}
		return pd.getReadMethod().getReturnType().getName();
	}

	private static String[] ignore = {};
	
	private void assignProperties(Object e, StartElement element) 
	{
		BeanWrapper wrapper = new BeanWrapperImpl(e);
		wrapper.registerCustomEditor(QName.class, new QNameEditor());
		
		for (Iterator<Attribute> itr = element.getAttributes(); itr.hasNext();)
		{
			Attribute attribute = itr.next();
			String name = attribute.getName().getLocalPart();
			if (ArrayUtils.contains(ignore, name))
			{
				continue;
			}
			
			Object value =  attribute.getValue();
			
			Class attributeClass = BeanUtils.findPropertyType(name, new Class[] { e.getClass() });
			if (attributeClass.isEnum())
			{
				Method method = BeanUtils.findDeclaredMethodWithMinimalParameters(attributeClass, "fromValue");
				try 
				{
					value = method.invoke(null, value);
				} 
				catch (IllegalArgumentException e1) 
				{
					log.error("IllegalArgumentException trying to invoke fromValue");
					return;
				} 
				catch (IllegalAccessException e1) 
				{
					log.error("IllegalAccessException trying to invoke fromValue");
					return;
				} 
				catch (InvocationTargetException e1) 
				{
					log.error("InvocationTargetException trying to invoke fromValue");
					return;
				}
			}
			PropertyDescriptor pd = BeanUtils.getPropertyDescriptor(e.getClass(), name);
			if (pd == null)
			{
				PropertyDescriptor[] pds = BeanUtils.getPropertyDescriptors(e.getClass());
				for (PropertyDescriptor pd1 : pds)
				{
					if (pd1.getName().equalsIgnoreCase(name))
					{
						name = pd1.getDisplayName();
						break;
					}
				}
			}
			try
			{
				if (name != null && name.startsWith("ea_"))
                {
                    name = name.replaceFirst("ea_", "");
                }

                wrapper.setPropertyValue(name, value);
			}
			catch (Exception ex)
			{
				log.info("Error writing property value:" + name + " " + e.getClass().getName());
			}
		}
	}

	public Map<String, Class<?>> getElementMap() 
	{
		return elementMap;
	}

	public void setElementMap(Map<String, Class<?>> elementMap) 
	{
		this.elementMap = elementMap;
	}

}
