package net.sf.javaprinciples.persistence.db.jdbc;

import java.util.List;

import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcOperations;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import net.sf.javaprinciples.persistence.db.jdbc.support.support.EnhancedBeanPropertySqlParameterSource;
import net.sf.javaprinciples.persistence.db.jdbc.support.support.NamedQueryProvider;
import net.sf.javaprinciples.persistence.db.jdbc.support.support.SqlParameterSourceFactory;

/**
 * Support class providing common functionality for Product and Task DAOs.
 * 
 * @author rvanluinen
 *
 */
public abstract class DaoSupport
{
    // Consider whether or not we want access to these in subclasses.
    protected NamedParameterJdbcOperations jdbcTemplate;
    protected NamedQueryProvider queryProvider;
    protected SqlParameterSourceFactory sqlParameterSourceFactory;
    private String primaryKeyColumnName = "row_id";

    protected DaoSupport(NamedParameterJdbcOperations jdbcTemplate,
                          NamedQueryProvider queryProvider, SqlParameterSourceFactory sqlParameterSourceFactory)
    {
        this.jdbcTemplate = jdbcTemplate;
        this.queryProvider = queryProvider;
        this.sqlParameterSourceFactory = sqlParameterSourceFactory;
    }

    protected void logException(String method, DataAccessException ex)
    {
        String cause = "";
        if (ex.getCause() != null)
        {
            cause = " - Cause: " + ex.getCause().getMessage();
        }
        String className = getClass().getName();
        // TODO - test compatibility of slf4j v1.7.1 so we can remove the array creation.
        LoggerFactory.getLogger(className).error("{} operation: {} {}{}",
                new Object[] {className, method, ex.getMessage(), cause});
    }

    /**
     * Insert a row.
     * @param parameterSource The columns to persist, must not be null
     * @param operation The query to obtain to facilitate the insert. Must not be null.
     * @return The primary key returned from the persistent store.
     */
    protected long performInsert(SqlParameterSource parameterSource, Enum<?> operation)
    {
        try
        {
            String query = queryProvider.getQuery(operation.name());
            KeyHolder generatedKeyHolder = new GeneratedKeyHolder();
            // Need to provide the column names
            // http://stackoverflow.com/questions/2819514/using-springs-keyholder-with-programmatically-generated-primary-keys
            jdbcTemplate.update(query, parameterSource, generatedKeyHolder, new String[] {primaryKeyColumnName});
            return generatedKeyHolder.getKey().longValue();
        }
        catch (DataAccessException e)
        {
            logException(operation.name(), e);
            throw e;
        }
    }

    protected <T> List<T> performQuery(SqlParameterSource parameterSource, Enum<?> operation, ParameterizedRowMapper<T> rowMapper)
    {
        try
        {
            String query = queryProvider.getQuery(operation.name());
            return jdbcTemplate.query(query, parameterSource, rowMapper);
        }
        catch (DataAccessException e)
        {
            logException(operation.name(), e);
            throw e;
        }
    }

    protected <T> T performQueryForZeroToSingleResult(SqlParameterSource parameterSource, Enum<?> operation, ParameterizedRowMapper<T> rowMapper)
    {
        try
        {
            String query = queryProvider.getQuery(operation.name());
            return jdbcTemplate.queryForObject(query, parameterSource, rowMapper);
        }
        catch (EmptyResultDataAccessException e)
        {
            return null;
        }
        catch (DataAccessException e)
        {
            logException(operation.name(), e);
            throw e;
        }
    }

    protected <T> T performQueryForSingleResult(SqlParameterSource parameterSource, Enum<?> operation, ParameterizedRowMapper<T> rowMapper)
    {
        try
        {
            String query = queryProvider.getQuery(operation.name());
            return jdbcTemplate.queryForObject(query, parameterSource, rowMapper);
        }
        catch (DataAccessException e)
        {
            logException(operation.name(), e);
            throw e;
        }
    }

    protected int performQueryForIntResult(SqlParameterSource parameterSource, Enum<?> operation)
    {
        try
        {
            String query = queryProvider.getQuery(operation.name());
            return jdbcTemplate.queryForInt(query, parameterSource);
        }
        catch (DataAccessException e)
        {
            logException(operation.name(), e);
            throw e;
        }
    }

    protected EnhancedBeanPropertySqlParameterSource getParameterSource()
    {
        return (EnhancedBeanPropertySqlParameterSource)sqlParameterSourceFactory.getSqlParameterSource();
    }

    protected EnhancedBeanPropertySqlParameterSource getParameterSource(Object sourceBean)
    {
        EnhancedBeanPropertySqlParameterSource parameterSource = getParameterSource();
        parameterSource.setBeanParameterSource(sourceBean);
        return parameterSource;
    }

    public void setPrimaryKeyColumnName(String primaryKeyColumnName)
    {
        this.primaryKeyColumnName = primaryKeyColumnName;
    }
}
