package net.sf.javaprinciples.persistence.db.jdbc.support.support;

import java.beans.PropertyDescriptor;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.NotWritablePropertyException;
import org.springframework.beans.PropertyAccessorFactory;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.convert.ConversionService;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.jdbc.core.simple.ParameterizedBeanPropertyRowMapper;
import org.springframework.jdbc.support.JdbcUtils;

/**
 * An extension of the Spring JDBC ParameterizedBeanPropertyRowMapper providing additional application-specific
 * extraction and conversion of SQL query results into Java object instances.
 * 
 * Specifically, this class provides two useful enhancements:
 * 
 * 1. Configures the conversionService property of BeanWrapper instances prior to their use in mapping columns from the
 * ResultSet into Bean properties of result objects. This allows custom, specific type conversions to be configured for
 * certain properties. One example of this is to take the (XML) contents of a CLOB and, using JAXB, unmarshall the XML
 * into the correct object prior to setting it on the result bean. Additional conversions can be added as required via
 * configuration.
 * 
 * 2. Allows the setting of nested bean properties to an arbitrary depth based on the column label / name returned
 * by a query. This allows for the population of an object graph directly from a query that may contain a
 * set of complicated joins, allowing the database to be used for its strengths. An example of this is to find the
 * latest version of a set of rows associated with a given entity via a foreign key relationship without having to
 * fetch all of the rows and then iterate through them from within java.
 * 
 * Such nested properties use the following syntax: propertyName_subPropertyName_subPropertyName and so on.
 * If a query such as this one is executed: select p.value as propertyName_subPropertyName from table aTable p
 * Then this mapper will attempt to perform the equivalent of result.getPropertyName().setSubPropertyName(value).
 * The BeanWrapper instance used for this purpose is set to autogrow nested paths so that any objects in the path not
 * yet constructed will be created and set, avoiding attempts to set values on null objects.
 * 
 * @author rvanluinen
 *
 * @param <T> The type of the result bean returned for each row.
 */
public class ConversionServiceAwareBeanPropertyRowMapper<T> extends ParameterizedBeanPropertyRowMapper<T>
{
    private ConversionService conversionService;
    // This cache is thread safe since we never actually use the cached wrappers for setting values on their
    // wrapped beans. They are only used to provide a mechanism to translate the property name extracted from
    // the query result into a correct JavaBean one. For example, property_subProperty in a SQL query will actually
    // appear as property_subproperty (or the same in all upper case) when the column label is looked-up.
    // The subproperty won't be found in this instance since there will be no setSubproperty() method on the bean,
    // it is actually setSubProperty()
    // It uses ConcurrentHashMap to alleviate potential problems with HashMap where it could
    // enter an infinite loop on concurrent reads/writes.
    private Map<Class<?>, BeanWrapper> beanWrapperCache;

    public ConversionServiceAwareBeanPropertyRowMapper(Class<T> mappedClass, ConversionService conversionService)
    {
        this.conversionService = conversionService;
        beanWrapperCache = new ConcurrentHashMap<Class<?>, BeanWrapper>();
        setMappedClass(mappedClass);
    }

    @Override
    public T mapRow(ResultSet rs, int rowNumber) throws SQLException
    {
        // Use the super class behaviour to map everything other than the nested properties we deal with.
        T rootObject = super.mapRow(rs, rowNumber);

        BeanWrapper bw = PropertyAccessorFactory.forBeanPropertyAccess(rootObject);
        bw.setAutoGrowNestedPaths(true);
        initBeanWrapper(bw);

        ResultSetMetaData rsmd = rs.getMetaData();
        int columnCount = rsmd.getColumnCount();

        for (int index = 1; index <= columnCount; index++)
        {
            String column = JdbcUtils.lookupColumnName(rsmd, index);
            if (isSubPropertyExpression(bw, column))
            {
                setSubPropertyValue(rs, bw, index, column);
            }
        }
        return rootObject;
    }

    private void setSubPropertyValue(ResultSet rs, BeanWrapper bw, int index, String column) throws SQLException
    {
        String propertyPath = makeSanePropertyPath(bw, column.toLowerCase());
        PropertyDescriptor pd = bw.getPropertyDescriptor(propertyPath);
        try
        {
            Object value = getColumnValue(rs, index, pd);
            try
            {
                bw.setPropertyValue(propertyPath, value);
            }
            catch (TypeMismatchException e)
            {
                if (value != null)
                {
                    throw e;
                }
            }
        }
        catch (NotWritablePropertyException ex)
        {
            throw new DataRetrievalFailureException(
                    "Unable to map column " + column + " to property " + pd.getName(), ex);
        }
    }

    private boolean isSubPropertyExpression(BeanWrapper bw, String column)
    {
        int idx = column.indexOf('_');
        if (idx < 1 || idx == column.length() - 1)
        {
            return false;
        }
        String propertyName = makeSaneProperty(bw, column.substring(0, idx).toLowerCase());
        return bw.isWritableProperty(propertyName);
    }

    private String makeSanePropertyPath(BeanWrapper bw, String propertyPath)
    {
        StringBuffer sanePath = new StringBuffer();
        boolean first = true;
        BeanWrapper currentBeanWrapper = bw;
        int componentIdx = 0;
        String[] pathComponents = propertyPath.split("_");
        for (String pathComponent : pathComponents)
        {
            String saneComponent = makeSaneProperty(currentBeanWrapper, pathComponent);
            if (componentIdx < pathComponents.length - 1)
            {
                currentBeanWrapper = getBeanWrapper(currentBeanWrapper, saneComponent);
            }
            if (!first)
            {
                sanePath.append(".");
            }
            sanePath.append(saneComponent);
            first = false;
            componentIdx++;
        }
        return sanePath.toString();
    }

    private BeanWrapper getBeanWrapper(BeanWrapper currentBeanWrapper, String saneComponent)
    {
        Class<?> componentType = currentBeanWrapper.getPropertyType(saneComponent);
        if (!beanWrapperCache.containsKey(componentType))
        {
            beanWrapperCache.put(componentType, new BeanWrapperImpl(componentType));
        }
        return beanWrapperCache.get(componentType);
    }

    private String makeSaneProperty(BeanWrapper bw, String propertyName)
    {
        for (PropertyDescriptor pd : bw.getPropertyDescriptors())
        {
            if (pd.getName().toLowerCase().equals(propertyName))
            {
                return pd.getName();
            }
        }
        return propertyName;
    }

    @Override
    protected void initBeanWrapper(BeanWrapper bw)
    {
        bw.setConversionService(conversionService);
    }
}
