package net.sf.javaprinciples.persistence.db.jdbc.support.support;

import java.sql.Types;
import java.util.HashMap;
import java.util.Map;

import org.springframework.core.convert.ConversionService;
import org.springframework.jdbc.core.namedparam.AbstractSqlParameterSource;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.core.support.SqlLobValue;
import org.springframework.jdbc.support.lob.LobHandler;

/**
 * A SqlParameterSource for use with Spring JDBC infrastructure that provides functionality combining
 * the capabilities of BeanPropertySqlParameterSource and MapSqlParameterSource, enhancing them with
 * additional custom type conversion capabilities and Clob support.
 * 
 * This class allows for an object to be used as a SqlParameterSource for matching properties
 * (BeanPropertySqlParameterSource functionality) and for additional properties to be specified via
 * MapSqlParameterSource functionality to cover cases where the object's properties are not sufficient. It also
 * allows for an object's property values to be overridden since the map properties take precedence.
 * 
 * The implementation allows for an optional ConversionService to be configured that will allow customised
 * conversion of parameter values prior to them being set. This has been used for example in marshalling an object
 * into XML using JAXB prior to inserting the value into the database.
 * 
 * @author rvanluinen
 *
 */
public class EnhancedBeanPropertySqlParameterSource extends AbstractSqlParameterSource
{
    private SqlParameterSource beanPropertySource;
    private MapSqlParameterSource additionalParameters;
    private Map<String, Integer> parameterToSqlType;
    private ConversionService conversionService;
    private LobHandler lobHandler;

    public EnhancedBeanPropertySqlParameterSource(ConversionService conversionService)
    {
        additionalParameters = new MapSqlParameterSource();
        parameterToSqlType = new HashMap<String, Integer>();
        this.conversionService = conversionService;
    }

    public void setLobHandler(LobHandler lobHandler)
    {
        this.lobHandler = lobHandler;
    }

    public void setBeanParameterSource(Object bean)
    {
        this.beanPropertySource = new BeanPropertySqlParameterSource(bean);
    }

    public void addParameter(String paramName, Object paramVal)
    {
        additionalParameters.addValue(paramName, paramVal);
    }

    public void setSqlTypeForParameter(String paramName, int sqlType)
    {
        parameterToSqlType.put(paramName, Integer.valueOf(sqlType));
    }

    @Override
    public boolean hasValue(String paramName)
    {
        if (additionalParameters.hasValue(paramName))
        {
            return true;
        }
        return beanHasValue(paramName);
    }

    @Override
    public Object getValue(String paramName) throws IllegalArgumentException
    {
        if (additionalParameters.hasValue(paramName))
        {
            return convertIfNecessary(paramName, additionalParameters.getValue(paramName));
        }
        return convertIfNecessary(paramName, beanValue(paramName));
    }

    private boolean beanHasValue(String paramName)
    {
        if (beanPropertySource != null)
        {
            return beanPropertySource.hasValue(paramName);
        }
        return false;
    }

    private Object beanValue(String paramName)
    {
        if (beanPropertySource != null)
        {
            return beanPropertySource.getValue(paramName);
        }
        return null;
    }

    private Object convertIfNecessary(String paramName, Object beanValue)
    {
        if (beanValue == null)
        {
            return beanValue;
        }
        Object converted = beanValue;
        if (conversionService != null && conversionService.canConvert(beanValue.getClass(), String.class))
        {
            converted = conversionService.convert(beanValue, String.class);
        }
        if (converted instanceof Enum)
        {
            return converted.toString();
        }
        if (getSqlType(paramName) == Types.CLOB)
        {
            return new SqlLobValue(converted.toString(), lobHandler);
        }
        return converted;
    }

    @Override
    public int getSqlType(String paramName)
    {
        if (parameterToSqlType.containsKey(paramName))
        {
            return parameterToSqlType.get(paramName).intValue();
        }
        if (additionalParameters.hasValue(paramName))
        {
            return additionalParameters.getSqlType(paramName);
        }
        if (beanPropertySource != null)
        {
            return beanPropertySource.getSqlType(paramName);
        }
        return SqlParameterSource.TYPE_UNKNOWN;
    }
}
