package net.sf.javaprinciples.web;

import java.io.IOException;
import java.security.Principal;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import com.sun.security.auth.UserPrincipal;

import net.sf.javaprinciples.core.RequestContext;
import net.sf.javaprinciples.core.UnexpectedException;

/**
 * A Servlet Filter to log the access and execution time of a URL.
 * 
 * @author Warwick Slade
 * 
 */
public class ServletRequestContextFilter implements Filter
{
    // A custom header that may contain a correlation identifier
    private static final String CORRELATION = "X-Correlation";
    // A standard header that may contain the principal of the request
    private static final String AUTHORIZATION = "Authorization";
    // A init parameter name to be used for performance logging
    private static final String LOGGER = "logger";

    private FilterConfig config;
    private Logger logger;

    /**
     * @param config  FilterConfig.
     * @throws ServletException - exception.
     */
    @Override
    public void init(FilterConfig config) throws ServletException
    {
        this.config = config;

        String loggerName = config.getInitParameter(LOGGER);
        if (!StringUtils.hasText(loggerName))
        {
            loggerName =  ServletRequestContextFilter.class.getPackage().getName();
        }
        logger = LoggerFactory.getLogger(loggerName);
    }

    /**
     * Reset variables
     */
    @Override
    public void destroy()
    {
        config = null;
        logger = null;
    }

    /**
     * @param request - ServletRequest
     * @param response - ServletResponse
     * @param chain - FilterChain
     * @throws java.io.IOException - exception
     * @throws ServletException - exception
     */
    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException
    {
        if (!(request instanceof HttpServletRequest))
        {
            throw new UnexpectedException("This servlet only works with HTTP requests!");
        }
        correlateRequest((HttpServletRequest)request);
        try
        {
            logAndChainRequest((HttpServletRequest) request, (HttpServletResponse) response, chain);
        }
        finally
        {
            // Clear out values put onto this thread.
            RequestContext.resetCorrelation();
            RequestContext.resetPrincipal();
        }
    }

    protected void logAndChainRequest(HttpServletRequest request, HttpServletResponse response, FilterChain chain)
            throws IOException, ServletException
    {
        // Some values for logging
        String uri = request.getRequestURI();

        // Place any correlation on the response
        String correlation = RequestContext.getCorrelation();
        if (correlation != null)
        {
            response.addHeader(CORRELATION, correlation);
        }

        // Record the start time
        long before = System.currentTimeMillis();

        // Pass on request
        chain.doFilter(request, response);

        // Determine the duration
        long after = System.currentTimeMillis();
        long duration = after - before;
        long memory = Runtime.getRuntime().freeMemory() / (1024 * 1024);
        long total = Runtime.getRuntime().totalMemory() / (1024 * 1024);
        long max = Runtime.getRuntime().maxMemory() / (1024 * 1024);

        // This logging is intended for recording performance data in a CSV file. It needs to work in unison
        // with the logger configuration and follows the pattern of 'subject,verb,duration'
        logger.info("{},{},{},{},{},{}", uri, request.getMethod(), duration,memory, total, max);
    }

    protected void correlateRequest(HttpServletRequest httpRequest)
    {
        String correlationId = httpRequest.getHeader(CORRELATION);
        if (!StringUtils.hasText(correlationId))
        {
            RequestContext.putCorrelation();
        }
        else
        {
            RequestContext.putCorrelation(correlationId);
        }

        Principal principal = httpRequest.getUserPrincipal();
        if (principal == null)
        {
            String user = httpRequest.getHeader(AUTHORIZATION);
            if (!StringUtils.hasText(user))
            {
                user = config.getFilterName();
            }
            principal = new UserPrincipal(user);
        }
        RequestContext.putPrincipal(principal);
    }
}
