package net.sf.sparta.springwebutils.queryloader.impl;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.HashMap;
import java.util.Map;

import net.sf.sparta.springwebutils.queryloader.QueryLoader;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;

/**
 * Loads Queries from files. 
 * 
 * It's necessary to inform scriptsFolder at the creation time 
 * 
 * HOW TO USE: Declare the bean into applicationContext.xml (or beans.xml)
 * 	
 * {@literal 
 *      <bean id="loader" class="com.cablevision.common.db.impl.SimpleQueryLoader">
 *          <constructor-arg name="scriptsFolder" value="<<SCRIPTS_FOLDER_NAME>>" />
 *      </bean>
 * }
 * 
 * @author Daniel Conde Diehl
 *
 */
public class FileQueryLoader implements QueryLoader, InitializingBean {

	private final Map<String,String> QUERY_CACHE = new HashMap<String,String>(50);

	private String scriptsFolder;
	
	/**
	 * Constructor, receives the path for the folder containing all the queries
	 * 
	 * @param scriptsFolder
	 */
	public FileQueryLoader(String scriptsFolder) {
		this.scriptsFolder = scriptsFolder;
	}
	
	public FileQueryLoader() {
		
	}
	
	public void setScriptsFolder(String scriptsFolder) {
		this.scriptsFolder = scriptsFolder;
	}
	
	@Override
	public String load(String queryName) {
		String query = QUERY_CACHE.get(queryName); 
		if (query == null){
			query = loadfromFromFile(queryName);
			QUERY_CACHE.put(queryName, query);
		}
		return query;
		
	}
	
	/**
	 * Loads the query from the informed path and adds to cache
	 * 
	 * @param queryName File name for the query to be loaded 
	 * @return requested Query,
	 * @throws IllegalStateException In case query was not found
	 */
	private String loadfromFromFile(String queryName) throws IllegalStateException {
		StringBuilder buffer = new StringBuilder(2048);
		try {
			InputStream is = getClass().getResourceAsStream(scriptsFolder + queryName + ".sql");
			LineNumberReader reader = new LineNumberReader(new InputStreamReader(is));
			while (reader.ready()) {
				buffer.append(reader.readLine());
				buffer.append(' ');
			}
		} catch (Exception e) {
			throw new IllegalStateException("couldn't load query " + scriptsFolder + queryName, e);
		}
		return buffer.toString();		
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		Assert.notNull(scriptsFolder, "scriptsFolder can't be null");
		
	}

}
