package net.sourceforge.squirrel_sql.plugins.refactoring.commands;

/*
 * Copyright (C) 2007 Daniel Regli & Yannick Winiger
 * http://sourceforge.net/projects/squirrel-sql
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.sql.SQLException;

import net.sourceforge.squirrel_sql.client.gui.db.ColumnListDialog;
import net.sourceforge.squirrel_sql.client.session.ISession;
import net.sourceforge.squirrel_sql.client.session.SQLExecuterTask;
import net.sourceforge.squirrel_sql.client.session.SessionUtils;
import net.sourceforge.squirrel_sql.fw.dialects.DatabaseObjectQualifier;
import net.sourceforge.squirrel_sql.fw.dialects.HibernateDialect;
import net.sourceforge.squirrel_sql.fw.dialects.UserCancelledOperationException;
import net.sourceforge.squirrel_sql.fw.gui.GUIUtils;
import net.sourceforge.squirrel_sql.fw.sql.IDatabaseObjectInfo;
import net.sourceforge.squirrel_sql.fw.sql.ITableInfo;
import net.sourceforge.squirrel_sql.fw.sql.TableColumnInfo;
import net.sourceforge.squirrel_sql.fw.util.StringManager;
import net.sourceforge.squirrel_sql.fw.util.StringManagerFactory;
import net.sourceforge.squirrel_sql.fw.util.log.ILogger;
import net.sourceforge.squirrel_sql.fw.util.log.LoggerController;
import net.sourceforge.squirrel_sql.plugins.refactoring.gui.AddAutoIncrementDialog;

public class AddAutoIncrementCommand extends AbstractRefactoringCommand
{
	/**
	 * Logger for this class.
	 */
	@SuppressWarnings("unused")
	private final ILogger s_log = LoggerController.createLogger(AddAutoIncrementCommand.class);

	/**
	 * Internationalized strings for this class
	 */
	private static final StringManager s_stringMgr =
		StringManagerFactory.getStringManager(AddAutoIncrementCommand.class);

	static interface i18n
	{
		String SHOWSQL_DIALOG_TITLE = s_stringMgr.getString("AddAutoIncrementCommand.sqlDialogTitle");
	}

	protected AddAutoIncrementDialog customDialog;

	private ColumnListDialog listDialog;

	private TableColumnInfo columnToModify;

	public AddAutoIncrementCommand(ISession session, IDatabaseObjectInfo[] dbInfo)
	{
		super(session, dbInfo);
	}

	/**
	 * @see net.sourceforge.squirrel_sql.plugins.refactoring.commands.AbstractRefactoringCommand#onExecute()
	 */
	@Override
	protected void onExecute() throws SQLException
	{
		ITableInfo selectedTable = (ITableInfo) _info[0];
		TableColumnInfo[] tableColumnInfos = _session.getMetaData().getColumnInfo(selectedTable);

		if (tableColumnInfos.length == 1)
		{
			columnToModify = tableColumnInfos[0];
			showCustomDialog();
		} else
		{
			listDialog = new ColumnListDialog(tableColumnInfos, ColumnListDialog.MODIFY_COLUMN_MODE);
			listDialog.addColumnSelectionListener(new ColumnListSelectionActionListener());
			listDialog.setLocationRelativeTo(SessionUtils.getOwningFrame(_session));
			listDialog.setSingleSelection();
			listDialog.setTableName(selectedTable.getSimpleName());
			listDialog.setVisible(true);
		}
	}

	/**
	 * @see net.sourceforge.squirrel_sql.plugins.refactoring.commands.AbstractRefactoringCommand#generateSQLStatements()
	 */
	@Override
	protected String[] generateSQLStatements() throws UserCancelledOperationException
	{
		String[] result = new String[0];

		if (_dialect.supportsAutoIncrement())
		{
			DatabaseObjectQualifier qualifier =
				new DatabaseObjectQualifier(columnToModify.getCatalogName(), columnToModify.getSchemaName());

			// TODO: Allow the user to specify the sequence name for dialects that don't natively support 
			// auto-increment columns without using sequence/trigger combo.
			final StringBuilder sequenceName = new StringBuilder();
			sequenceName.append(columnToModify.getTableName().toUpperCase()).append("_");
			sequenceName.append(columnToModify.getColumnName().toUpperCase()).append("_SEQ");
			
			result =
				_dialect.getAddAutoIncrementSQL(columnToModify, sequenceName.toString(), qualifier, _sqlPrefs);
		} else
		{
			_session.showMessage(s_stringMgr.getString("AddAutoIncrementCommand.unsupportedOperationMsg",
				_dialect.getDisplayName()));
		}
		return result;
	}

	/**
	 * @see net.sourceforge.squirrel_sql.plugins.refactoring.commands.AbstractRefactoringCommand#executeScript(java.lang.String)
	 */
	@Override
	protected void executeScript(String script)
	{
		CommandExecHandler handler = new CommandExecHandler(_session);

		SQLExecuterTask executer = new SQLExecuterTask(_session, script, handler);
		executer.run(); // Execute the sql synchronously

		_session.getApplication().getThreadPool().addTask(new Runnable()
		{
			public void run()
			{
				GUIUtils.processOnSwingEventThread(new Runnable()
				{
					public void run()
					{
						customDialog.setVisible(false);
						customDialog.dispose();
						_session.getSchemaInfo().reloadAll();
					}
				});
			}
		});
	}

	/**
	 * Returns a boolean value indicating whether or not this refactoring is supported for the specified
	 * dialect.
	 * 
	 * @param dialectExt
	 *           the HibernateDialect to check
	 * @return true if this refactoring is supported; false otherwise.
	 */
	@Override
	protected boolean isRefactoringSupportedForDialect(HibernateDialect dialectExt)
	{
		return dialectExt.supportsAutoIncrement();
	}

	private void showCustomDialog()
	{
		_session.getApplication().getThreadPool().addTask(new Runnable()
		{
			public void run()
			{
				customDialog = new AddAutoIncrementDialog(columnToModify, SessionUtils.getOwningFrame(_session));
				customDialog.addExecuteListener(new ExecuteListener());
				customDialog.addEditSQLListener(new EditSQLListener(customDialog));
				customDialog.addShowSQLListener(new ShowSQLListener(i18n.SHOWSQL_DIALOG_TITLE, customDialog));
				customDialog.setLocationRelativeTo(SessionUtils.getOwningFrame(_session));
				customDialog.setVisible(true);
			}
		});
	}

	private class ColumnListSelectionActionListener implements ActionListener
	{
		public void actionPerformed(ActionEvent e)
		{
			if (listDialog == null)
			{
				System.err.println("dialog was null");
				return;
			}
			listDialog.setVisible(false);
			listDialog.dispose();

			columnToModify = listDialog.getSelectedColumnList()[0];
			showCustomDialog();
		}
	}
}
