/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.basher;

import java.util.Date;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class Average
{
    private long _minTime;
    private long _maxTime;
    private long _totalTime;
    private long _totalEntries;
    private long _totalSuccessful;
    private long _totalFailed;
    private long _totalNotRun;
    private long _timeElapsed;
    private int _numThreads;
    private Date _timeStamp;

    public Average(final long minTime, final long maxTime, final long totalTime, final long timeElapsed, final long totalSuccessful, final long totalFailed, final long totalNotRun, final int numThreads)
    {
        _minTime = minTime;
        _maxTime = maxTime;
        _totalTime = totalTime;
        _totalSuccessful = totalSuccessful;
        _totalFailed = totalFailed;
        _totalNotRun = totalNotRun;

        _totalEntries = (totalFailed + totalSuccessful);

        _timeElapsed = timeElapsed;
        _numThreads = numThreads;
        _timeStamp = new Date();
    }

    public long getTimeElapsed()
    {
        return _timeElapsed;
    }

    public long getAverage()
    {
        if (_totalEntries == 0)
        {
            return 0;
        }
        return _totalTime / _totalEntries;
    }

    public long getMinTime()
    {
        return _minTime;
    }

    public long getMaxTime()
    {
        return _maxTime;
    }

    public long getTotalTime()
    {
        return _totalTime;
    }

    public long getTotalEntries()
    {
        return _totalEntries;
    }

    public Date getTimeStamp()
    {
        return new Date(_timeStamp.getTime());
    }

    public float getTPS()
    {
        final float seconds = (_timeElapsed / 1000);
        if (seconds == 0)
        {
            return 0f;
        }
        return _totalEntries / seconds;
    }

    public int getNumThreads()
    {
        return _numThreads;
    }

    public long getTotalSuccessful()
    {
        return _totalSuccessful;
    }

    public long getTotalFailed()
    {
        return _totalFailed;
    }

    public long getTotalNotRun()
    {
        return _totalNotRun;
    }

    public String formatOutput()
    {
        return String.format( "elapsed time: %d min time: %s max time: %s total time: %s invocations: %d (%dS/%dF) not run: %d average time: %d tps: %f num threads: %d",
                _timeElapsed, (_minTime == Long.MAX_VALUE ? "N/A" : _minTime), _maxTime,_totalTime,
                _totalEntries,_totalSuccessful, _totalFailed, _totalNotRun, getAverage(),getTPS(),getNumThreads()
        );
    }

    public String toString()
    {
        return "elapsed time: " + _timeElapsed + " min time: " + (_minTime == Long.MAX_VALUE ? "N/A" : "" + _minTime)
                + " max time: " + _maxTime + " total time: " + _totalTime + " invocations: " + _totalEntries + " average time: "
                + getAverage() + " tps: " + getTPS() + " num threads: " + getNumThreads();
    }

    @Override
    public boolean equals( final Object o )
    {
        if ( this == o )
        {
            return true;
        }
        if ( o == null || getClass() != o.getClass() )
        {
            return false;
        }

        final Average average = ( Average ) o;

        if ( _maxTime != average._maxTime )
        {
            return false;
        }
        if ( _minTime != average._minTime )
        {
            return false;
        }
        if ( _numThreads != average._numThreads )
        {
            return false;
        }
        if ( _timeElapsed != average._timeElapsed )
        {
            return false;
        }
        if ( _totalEntries != average._totalEntries )
        {
            return false;
        }
        if ( _totalFailed != average._totalFailed )
        {
            return false;
        }
        if ( _totalNotRun != average._totalNotRun )
        {
            return false;
        }
        if ( _totalSuccessful != average._totalSuccessful )
        {
            return false;
        }
        if ( _totalTime != average._totalTime )
        {
            return false;
        }
        if ( _timeStamp != null ? !_timeStamp.equals( average._timeStamp ) : average._timeStamp != null )
        {
            return false;
        }

        return true;
    }
}
