/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher;

import java.util.Iterator;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;

import bsh.EvalError;
import bsh.Interpreter;
import org.ops4j.gaderian.Registry;
import org.ops4j.gaderian.impl.RegistryBuilder;

/**
 * Main entry point to the Basher when using the command line mode or for applications not using Gaderian.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public class Basher
{
    /**
     * Basher Gaderian registry instance
     */
    private static Registry _registry;
    /**
     * Shutdown thread instance
     */
    private Thread _hook;
    /**
     * Flag indicating if looping should continue
     */
    private boolean _keepRunning = true;

    /**
     * Flag indicating whether or not a shutdown has been initiated
     */
    private static AtomicBoolean _shutdownInitiated = new AtomicBoolean(false);

    /**
     * Main method for initializing and starting the <code>Basher</code> system.
     * This method accepts the following parameters:
     * -p port-number : specifies a port on which a bean shell server should be started.
     *
     * @param args The arguments to the program
     */
    public static void main(final String[] args)
    {
        final Basher basher = new Basher();

        // Add a hook to shut the system down correctly
        basher.createShutdownHook();

        // Add the timed shutdown if args says so
        basher.startShutdownTimer(args);

        // Set any system propertues
        basher.setSystemParameters(args);

        // Load up the system - this will also boot the Gaderian registry
        basher.startIt();

        // This will check to see if a port is specified and if so
        // start beanshell on that particular port...
        basher.startBeanshell(args);

        // And loop forever to keep the system running
        basher.loop();

    }

    /**
     * Extract and set any system parameters that may have been passed ont he command line.
     * The format of the parameter must be -Dsomekey=somevalue
     *
     * @param args The arguments to parse
     */
    private void setSystemParameters(final String[] args)
    {

        if (args.length != 0)
        {
            for (int i = 0; i < args.length; i++)
            {
                final String arg = args[i];
                if (arg.startsWith("-D"))
                {
                    final String[] properties = arg.substring(2).split("=");
                    if (properties.length != 2)
                    {
                        System.out.format("Invalid system property: %s" , arg.substring(2));
                        System.out.println("Format should be '-Dproperty=value");
                        usage();
                    }
                    System.setProperty(properties[0], properties[1]);
                }
            }
        }

    }

    /**
     * Checks and if required starts a shutdown timer that will shut down the Basher after a number of seconds.
     * The command line flag is -runfor N (where N is specified in seconds).
     *
     * @param args The arguments to parse
     */
    private void startShutdownTimer(final String[] args)
    {

        if (args.length != 0)
        {
            final int index = findIndexOf(args, "-runfor");

            if (index == -1)
            {
                return;
            }


            long runFor = 0;
            try
            {
                runFor = Long.parseLong(args[index + 1]);
            }
            catch (NumberFormatException e)
            {
                System.out.format("invalid value for seconds: %s" , args[index + 1]);
                usage();
            }

            final long runForReal = runFor;

            System.out.format("Scheduling timed shutdown after %d seconds", runForReal);

            if (_hook != null)
            {
                System.out.println("Removing registered shutdown hook");
                Runtime.getRuntime().removeShutdownHook(_hook);
            }
            final Timer timer = new Timer(true);
            timer.schedule(new TimerTask()
            {
                public void run()
                {
                    shutdown("Shutting down after " + runForReal + " seconds of running");
                }
            }, runFor * 1000);


        }

    }

    /**
     * Starts the Basher by constructing the default Gaderian registry.
     */
    private void startIt()
    {
        if (_registry == null)
        {
            setRegistry(RegistryBuilder.constructDefaultRegistry());
        }
        final Scheduler scheduler = (Scheduler)_registry.getService(Scheduler.class);
        scheduler.start();
    }

    /**
     * Loop which will ensure we keep running.  There may be better ways of doing this.
     */
    private void loop()
    {
        while (_keepRunning)
        {
            try
            {
                Thread.sleep(1000);
            }
            catch (InterruptedException e)
            {
                // what to do? what to doooo ... ;)
            }
        }
    }

    /**
     * Creates and install a shutdown hook (thread) to catch any shutdown instructions.
     */
    private void createShutdownHook()
    {
        _hook = new Thread(new Runnable()
        {
            public void run()
            {
                shutdown("Shutting down due to user interrupt");
            }
        });
        Runtime.getRuntime().addShutdownHook(_hook);
    }

    /**
     * Shuts the Basher down.  This will ensure the scheduler is stopped, wait until all threads have stopped and then
     * shut the Gaderian registry down.
     *
     * @param shutdownMessage The message to display when shutting down.
     */
    private void shutdown(final String shutdownMessage)
    {
        // First check if shutdown has been initiated
        if (_shutdownInitiated.getAndSet(true))
        {
            System.out.println("Shutdown already initiated, ignoring call");
            return;
        }

        System.out.println(shutdownMessage);

        // Shut down the system
        if (_registry != null)
        {
            final Scheduler scheduler = (Scheduler) _registry.getService(Scheduler.class);
            if (scheduler != null && scheduler.isRunning())
            {
                scheduler.stop();
            }
            System.out.println("Cooling down ...");
            boolean allDone = false;
            while (!allDone)
            {
                final Thread[] threadList = new Thread[Thread.activeCount()];
                final int numThreads = Thread.currentThread().getThreadGroup().enumerate(threadList);
                allDone = true;
                for (int j = 0; j < numThreads; j++)
                {
                    final Thread thread = threadList[j];
                    if (thread != null)
                    {
                        if (thread.getName().startsWith("Basher"))
                        {
                            allDone = false;
                            break;
                        }
                    }
                }
                try
                {
                    Thread.sleep(300);
                }
                catch (InterruptedException e)
                {
                    // Safe to ignore?
                }
            }


            System.out.println("Task Invocation Statistics (total/successes/failures)");

            final TaskManager taskManager = (TaskManager) _registry.getService(TaskManager.class);
            if (taskManager != null)
            {
                if (taskManager.getTasks().size() == 0)
                {
                    System.out.println("No tasks invoked");
                }
                System.out.println("Active Tasks");
                for ( final Task task : taskManager.getTasks() )
                {
                    System.out.format( "%s (%d/%d/%d)", task.getName(), task.getInvocations(), task.getSuccesses(), task.getFailures() );
                }
                System.out.println("Removed Tasks");
                for ( final Task task : taskManager.getRemovedTasks() )
                {
                    System.out.format( "%s (%d/%d/%d)", task.getName(), task.getInvocations(), task.getSuccesses(), task.getFailures() );
                }
            }

            System.out.println("Shutting down registry ...");
            _registry.shutdown();

            try
            {
                Thread.sleep(1000);
            }
            catch (InterruptedException e)
            {
                // Safe to ignore?
            }
        }
        _keepRunning = false;

    }

    /**
     * Check and if required start the BeanShell console.  The command line flag is -p N (where N specifies a port number
     * above 1024)
     *
     * @param args The arguments to parse
     */
    private void startBeanshell(final String[] args)
    {
        if (args.length != 0)
        {
            final int portIndex = findIndexOf(args, "-p");
            if (portIndex == -1)
            {
                return;
            }
            int port = 1234;

            try
            {
                port = Integer.parseInt(args[portIndex + 1]);
            }
            catch (NumberFormatException e)
            {
                System.out.format("invalid port: %s", args[portIndex + 1]);
                usage();
            }

            if (port < 1024)
            {
                System.out.println("port number must be greater than 1024");
                usage();
            }

            final Interpreter interpreter = new Interpreter();

            try
            {
                interpreter.set("registry", _registry);
                interpreter.set("taskManager", _registry.getService(TaskManager.class));
                interpreter.set("scheduler", _registry.getService(Scheduler.class));
            }
            catch (EvalError evalError)
            {
                evalError.printStackTrace();
            }

            try
            {
                interpreter.set("portnum", port);
                interpreter.eval("setAccessibility(true)"); // turn off access restrictions

                interpreter.eval("server(portnum)");
                System.out.format("beanshell server started on port %d", port);
            }
            catch (Throwable throwable)
            {
                System.out.println("error starting the beanshell server");
            }
        }
    }

    /**
     * Extracts the index of a command line flag with the specified name.  The command line flag to be extracted is
     * expected to have a parameter.
     *
     * @param args The arguments to parse
     * @param name The name of the argument to find
     * @return The index of the argument, or -1 if not found.
     */
    private int findIndexOf(final String[] args, final String name)
    {
        for (int i = 0; i < args.length; i++)
        {
            final String arg = args[i];
            if (arg.equals(name))
            {
                if ((i + 1) == args.length)
                {
                    usage();
                }
                return i;
            }
        }
        return -1;
    }

    /**
     * Display a usage message.
     */
    private void usage()
    {
        System.out.println("usage: basher [-p port] [-runfor minutes] [-Dproperty=value]");
        System.exit(1);
    }

    /**
     * Sets the registry this Basher instance should use.
     *
     * @param registry The Gaderian registry to use.
     * @throws IllegalArgumentException If the registry has already been set.
     */
    public static void setRegistry(final Registry registry)
    {
        if (_registry != null)
        {
            throw new IllegalStateException("Basher has already been initialized");
        }
        _registry = registry;
    }

    /**
     * Retrieves the registry used by Basher
     *
     * @return The registry in use
     * @throws IllegalStateException If the registry has not been set.
     */
    public static Registry getRegistry()
    {
        if (_registry == null)
        {
            throw new IllegalStateException("Basher has not been initialized");
        }
        if (_shutdownInitiated.get())
        {
            throw new IllegalStateException("Basher shutdown has been initiated");
        }
        return _registry;
    }

    /**
     * Convenience method to be used when a thread cleanup of the registry should happen.
     */
    public static void fireCleanUpThread()
    {
        if (_shutdownInitiated.get())
        {
            return;
        }
        try
        {
            if (_registry != null)
            {
                _registry.cleanupThread();
            }
        }
        catch (Throwable e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Retrieves a service from the Basher (Gaderian) registry.  This is only to be used by
     * application that do not themselves use Gaderian.
     *
     * @param service The class of the service to retrieve.
     * @return An instance of the service.
     * @throws Exception If the registry fails to intialize or the service does not exist.
     */
    public static <T> T getService(final Class<T> service) throws Exception
    {
        if (_registry == null)
        {
            _registry = RegistryBuilder.constructDefaultRegistry();
        }
        return (T)_registry.getService(service);
    }

}
