/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher;

import java.util.List;

/**
 * Defines the operations required for collecting statistical information about the running tasks.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public interface Collector
{
    /**
     * Signals to the collector that statistics collection should start.
     */
    public void startCollecting();

    /**
     * Signals to the collector that collection should stop.
     */
    public void stopCollecting();

    /**
     * Checks if the collector is currently collecting statistics.
     *
     * @return True if the collector is collecting data, otherwise false.
     */
    public boolean isCollecting();

    /**
     * Reports a task's successful invocation together with the time elapsed for the task to run.
     *
     * @param task        The task that was invoked
     * @param elapsedTime The time it took for the task to run
     */
    public void success(Task task, final long elapsedTime);

    /**
     * Reports a task's failure to run together with the time time elapsed for the failure.  The (possible) cause of the
     * failure is also provided.
     *
     * @param task        The task that failed
     * @param elapsedTime The time it took for the task to run and fail.
     * @param throwable   The cause of the failure.  This may be null if the cause of the failure is not an exception.
     */
    public void fail(Task task, final long elapsedTime, Throwable throwable);

    /**
     * Retrieves the total number of failures recorded.
     *
     * @return The number of failures recorded.
     */
    public long getFailures();

    /**
     * Retrieves the total number of successes recorded.
     *
     * @return The number of successes recorded.
     */
    public long getSuccesses();

    /**
     * Retrieves the total number of tasks that were recorded (essentially successes+failures).
     *
     * @return The total number of tasks recorded.
     */
    public long getTotal();

    /**
     * Informs the collector that a collection period has finished.
     */
    public Average markAverage();

    /**
     * Retrieves the list of averages calculated over time.
     *
     * @return A list of <code>Average</code> instances.
     */
    public List<Average> getAverages();

    /**
     * Reports a task's wish to not run altogether with the time time elapsed for it to determine this.
     *
     * @param task        The task that didn't wish to run
     * @param elapsedTime The time it took for the task to run and determine it didn't want to run
     */
    public void notRun(Task task, long elapsedTime);

    /**
     * Retrieves the total number of tasks that didn't wish to run
     *
     * @return The total number of tasks not wishing to run.
     */
    public long getNotRun();
}
