/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.impl;

import java.io.*;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;

import net.sourceforge.basher.Average;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.BasherContext;
import org.ops4j.gaderian.events.RegistryShutdownListener;

/**
 * @author Johan Lindquist
 * @version $Revision$
 */
public abstract class AbstractFileCollector extends AbstractCollector implements RegistryShutdownListener
{
    protected final Map<String, OpenFile> _openFiles = new HashMap<String, OpenFile>();
    protected File _parent;
    protected String _collectionDirectory;

    private String _averageHeader = null;
    private String _averageFooter = null;
    private String _executionHeader = null;
    private String _executionFooter = null;

    private String _averageFilenamePrefix = "averages.";
    private String _extension;

    private int _queueCapacity = 200;
    private BlockingQueue<FileEntry> _fileEntries;
    private int _numThreads = 5;
    private List<FileEntryProcessor> _fileEntryProcessors;
    private long _timeOut = 500;
    private boolean _keepRunning = true;
    private boolean _ready = true;

    public void setNumThreads(final int numThreads)
    {
        _numThreads = numThreads;
    }

    public void setTimeOut(final long timeOut)
    {
        _timeOut = timeOut;
    }

    public void setQueueCapacity(final int queueCapacity)
    {
        _queueCapacity = queueCapacity;
    }

    public void setAverageFooter(final String averageFooter)
    {
        _averageFooter = averageFooter;
    }

    public void setExecutionFooter(final String executionFooter)
    {
        _executionFooter = executionFooter;
    }

    public void setAverageHeader(final String averageHeader)
    {
        _averageHeader = averageHeader;
    }

    public void setExecutionHeader(final String executionHeader)
    {
        _executionHeader = executionHeader;
    }

    public void setExtension(final String extension)
    {
        _extension = extension;
    }

    public void setAverageFilenamePrefix(final String averageFilenamePrefix)
    {
        _averageFilenamePrefix = averageFilenamePrefix;
    }

    /**
     * Initializes the service.
     *
     * @throws Exception If the script directory does not exist or can not be written to
     */
    public void initializeService() throws Exception
    {

    }

    /**
     * Writes the result of a successful task execution to the success file.
     *
     * @param task        The task that executed successfully.
     * @param elapsedTime Time of execution
     */
    public void success(final Task task, final long elapsedTime)
    {
        super.success(task, elapsedTime);

        if (_ready && isCollecting())
        {
            writeToFile(Type.SUCCESS, task.getName(), elapsedTime);
        }
    }

    /**
     * Writes the result of a failed task to the failure file.
     *
     * @param task        The task that failed.
     * @param elapsedTime Time of execution
     * @param throwable   The reason (if specified) for the failure
     */
    public void fail(final Task task, final long elapsedTime, final Throwable throwable)
    {
        super.fail(task, elapsedTime, throwable);
        if (_ready && isCollecting())
        {
            writeToFile(Type.FAILURE, task.getName(), elapsedTime);
        }
    }

    protected void initializeCollector(final BasherContext basherContext) throws Exception
    {
        try
        {
            _collectionDirectory = basherContext.getReportDirectory();

            _parent = new File(_collectionDirectory);

            if (!_parent.exists())
            {
                if (!_parent.mkdir())
                {
                    throw new Exception("Could not create root collection directory '" + _collectionDirectory + "'");
                }
            }
            if (!_parent.canWrite())
            {
                throw new Exception("Root collection directory '" + _collectionDirectory + "' can not be written to");
            }

            /*// Ok, the directory is available and writable
            _parent = new File(_parent, "" + _timeSource.getStartTime());

            if (!_parent.exists())
            {
                if (!_parent.mkdir())
                {
                    throw new Exception("Could not create collection directory '" + _parent.getAbsolutePath() + "'");
                }
            }*/

            _log.info("Initializing queue with capacity: " + _queueCapacity);
            _fileEntries = new ArrayBlockingQueue<FileEntry>(_queueCapacity);

            _fileEntryProcessors = new ArrayList<FileEntryProcessor>(_numThreads);

            for (int i = 0; i < _numThreads; i++)
            {
                final FileEntryProcessor fileEntryProcessor = new FileEntryProcessor();
                fileEntryProcessor.setName("FileEntryProcessor_" + i);
                fileEntryProcessor.setDaemon(true);
                fileEntryProcessor.start();
                _fileEntryProcessors.add(fileEntryProcessor);
            }

            _log.debug("AbstractFileCollector initialized");
            _ready = true;
        }
        catch (Exception e)
        {
            _log.error("Error initializing: " +e.getMessage(),e);
            _log.error("Will not use filesystem");
            _ready = false;
        }
    }

    /**
     * Writes the result of a not tun task execution to the notrun file.
     *
     * @param task        The task that didn't run.
     * @param elapsedTime Time of execution
     */
    public void notRun(final Task task, final long elapsedTime)
    {
        super.notRun(task, elapsedTime);
        if (_ready &&isCollecting())
        {
            writeToFile(Type.NOT_RUN, task.getName(), elapsedTime);
        }
    }

    /**
     * Writes the specified <code>Average</code> to the averag file.
     *
     * @param average The average to write to disk
     */
    protected void dumpAverage(final Average average)
    {
        try
        {
            final String fileKey = _averageFilenamePrefix + _extension;
            BufferedWriter bw = getWriter(fileKey, Classifier.AVERAGE);
            bw.write(formatAverage(average));
            bw.flush();
        }
        catch (IOException e)
        {
            _log.error(e.getMessage(), e);
        }

    }

    protected abstract String formatAverage(final Average average);

    protected abstract String formatExecution(final String taskName, final long elapsedTime);

    /**
     * Writes the time the specified task took to execute.
     *
     * @param taskName    The name of the task (used for the filename as well)
     * @param type
     * @param elapsedTime Time the task took to execute
     */
    protected void writeToFile(final Type type, final String taskName, final long elapsedTime)
    {
        try
        {
            final String fileKey = getFileName(taskName, type);
            final String formattedText = formatExecution(taskName, elapsedTime);
            _fileEntries.put(new FileEntry(taskName, formattedText, fileKey, Classifier.EXECUTION));
        }
        catch (Exception e)
        {
            _log.error(e.getMessage(), e);
        }
    }

    private String getFileName(final String taskName, final Type type)
    {
        switch (type)
        {
            case FAILURE:
                return taskName + "-failed." + _extension;
            case NOT_RUN:
                return taskName + "-notrun." + _extension;
            case SUCCESS:
                return taskName + "-success." + _extension;
            default:
                throw new IllegalStateException("Unhandled type: " + type);
        }
    }

    /**
     * Ensures the average over the last time slice is written to disk.
     * {@inheritDoc}
     *
     * @return The average over the last time slice.
     */
    public Average markAverage()
    {
        final Average average = super.markAverage();
        if (_ready && isCollecting())
        {
            dumpAverage(average);
        }
        return average;
    }


    /**
     * Retrieves (and possibly creates) a new writer on to which to write data
     *
     * @param fileKey    The identifier of the buffer
     * @param classifier
     * @return The buffer
     * @throws java.io.IOException If the retrieval (create) fails
     */
    protected BufferedWriter getWriter(final String fileKey, final Classifier classifier, Object... headerParams)
            throws IOException
    {
        AbstractFileCollector.OpenFile openFile = _openFiles.get(fileKey);
        if (openFile == null)
        {
            synchronized (_openFiles)
            {
                openFile = _openFiles.get(fileKey);
                if (openFile == null)
                {
                    final BufferedWriter bw = new BufferedWriter(new FileWriter(new File(_parent, fileKey)));
                    openFile = new OpenFile(bw, classifier);
                    addHeader(bw, classifier, headerParams);
                    bw.flush();
                    _openFiles.put(fileKey, openFile);
                }
            }
        }
        return openFile._bufferedWriter;
    }

    public void registryDidShutdown()
    {
        _log.debug("Shutting down collector");
        stopCollecting();

        _log.debug("Collection queue size: " + _fileEntries.size());
        if (_fileEntries.size() > 0)
        {
            _log.debug("Draining collection queue");
            while (_fileEntries.size() > 0)
            {
                try
                {
                    Thread.sleep(250);
                }
                catch (InterruptedException e)
                {
                    // Safe to ignore
                }
            }
        }

        _keepRunning = false;

        _log.debug("Closing open files");
        _log.debug("Files open: " + _openFiles.size());
        
        final Collection<OpenFile> entries = _openFiles.values();
        for (final OpenFile entry : entries)
        {
            try
            {
                final Writer writer = entry._bufferedWriter;
                addFooter(entry._bufferedWriter, entry._classifier);
                writer.flush();
                writer.close();
            }
            catch (IOException e)
            {
                _log.error(e.getMessage(), e);
            }
        }
        _log.debug("Open files closed");
    }

    private void addFooter(final BufferedWriter bufferedWriter, final Classifier classifier) throws IOException
    {
        switch (classifier)
        {
            case AVERAGE:
                if (_averageFooter != null)
                {
                    bufferedWriter.write(_averageFooter);
                }
                break;
            case EXECUTION:
                if (_executionFooter != null)
                {
                    bufferedWriter.write(_executionFooter);
                }
                break;
            default:
                throw new IllegalStateException("Unhandled classifier: " + classifier);
        }
    }

    private void addHeader(final BufferedWriter bufferedWriter, final Classifier classifier, Object... headerParams) throws IOException
    {
        switch (classifier)
        {
            case AVERAGE:
                if (_averageHeader != null)
                {
                    bufferedWriter.write(String.format(_averageHeader,headerParams));
                }
                break;
            case EXECUTION:
                if (_executionHeader != null)
                {
                    if (headerParams == null || headerParams.length == 0)
                    {
                        bufferedWriter.write(_executionHeader);
                    }
                    else
                    {
                        bufferedWriter.write(String.format(_executionHeader,headerParams));
                    }
                }
                break;
            default:
                throw new IllegalStateException("Unhandled classifier: " + classifier);
        }
    }

    private class OpenFile
    {
        private final BufferedWriter _bufferedWriter;
        private final Classifier _classifier;

        public OpenFile(final BufferedWriter bufferedWriter, final Classifier classifier)
        {
            _bufferedWriter = bufferedWriter;
            _classifier = classifier;
        }

    }

    public class FileEntry
    {
        String _taskName;
        String _formattedText;
        String _fileKey;
        Classifier _classifier;

        public FileEntry(final String taskName, final String formattedText, final String fileKey, final Classifier classifier)
        {
            _taskName = taskName;
            _formattedText = formattedText;
            _fileKey = fileKey;
            _classifier = classifier;

        }
    }

    public static enum Classifier
    {
        AVERAGE, EXECUTION
    }

    public static enum Type
    {
        SUCCESS, FAILURE, NOT_RUN
    }

    private class FileEntryProcessor extends Thread
    {


        public void run()
        {
            while (_keepRunning)
            {
                try
                {
                    // Consume first entry
                    final AbstractFileCollector.FileEntry fileEntry = _fileEntries.poll(_timeOut, TimeUnit.MILLISECONDS);

                    if (fileEntry != null)
                    {
                        // Write it to the file
                        final BufferedWriter bw = getWriter(fileEntry._fileKey, fileEntry._classifier,fileEntry._taskName);
                        bw.write(fileEntry._formattedText);
                        bw.flush();
                    }
                }
                catch (Exception e)
                {
                    _log.error(e.getMessage(), e);
                }
            }
        }
    }
}
