/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.impl;

import java.util.ArrayList;
import java.util.List;

import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.ContextManager;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.events.BasherEvent;
import net.sourceforge.basher.events.BasherEventListener;
import net.sourceforge.basher.events.PhaseTransitionEvent;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 * @version $Revision$
 */
public class ContextManagerImpl implements ContextManager, BasherEventListener
{
    private Log _log;
    private BasherContext _activeBasherContext;
    private List<BasherContext> _basherContexts = new ArrayList<BasherContext>();

    /**
     * {@inheritDoc}
     */
    public BasherContext getActiveBasherContext()
    {
        return _activeBasherContext;
    }

    /**
     * {@inheritDoc}
     */
    public void setActiveBasherContext(final BasherContext basherContext)
    {
        _log.debug("Setting active basher context: " + basherContext);
        _activeBasherContext = basherContext;
    }

    public BasherContext getBasherContext(final String contextName)
    {
        _log.debug("Retrieving Bascher context by name: " + contextName);

        for (BasherContext basherContext : _basherContexts)
        {
            if (basherContext.getName().equals(contextName))
            {
                return basherContext;
            }
        }
        throw new IllegalStateException("No such context: " + contextName);
    }

    /**
     * {@inheritDoc}
     */
    public List<BasherContext> getBasherContexts()
    {
        return _basherContexts;
    }

    /**
     * {@inheritDoc}
     */
    public void addBasherContext(final BasherContext basherContext)
    {
        if (_basherContexts == null)
        {
            _basherContexts = new ArrayList<BasherContext>();
        }
        _basherContexts.add(basherContext);
    }

    /**
     * Sets the list of available <code>SystemContext</code>s.  The specified list will be added
     * the list of already present contexts if any.
     *
     * @param basherContexts The list of avialable system contexts.
     */
    public void setBasherContexts(final List<BasherContext> basherContexts)
    {
        if (_basherContexts == null)
        {
            _basherContexts = basherContexts;
        }
        else
        {
            _basherContexts.addAll(basherContexts);
        }
    }

    public void setLog(final Log log)
    {
        _log = log;
    }

    public void basherEvent(final BasherEvent basherEvent)
    {
        if (basherEvent instanceof PhaseTransitionEvent)
        {
            final PhaseTransitionEvent phaseTransitionEvent = (PhaseTransitionEvent) basherEvent;
            if (phaseTransitionEvent.getNewPhase() == Phase.SETUP)
            {
                _activeBasherContext = phaseTransitionEvent.getBasherContext();
            }
        }
    }
}
