/*
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.basher.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Arrays;

import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskManager;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.events.*;
import net.sourceforge.basher.internal.Randomizer;
import net.sourceforge.basher.internal.TaskCreator;
import net.sourceforge.basher.internal.TaskDecorator;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TaskManagerImpl implements TaskManager, BasherEventListener
{
    private List<TaskInfo> _taskInfos;
    private List<Task> _tasks;
    private List<Task> _availableTasks;
    private Randomizer _randomizer;
    private List<Task> _removedTasks;
    private Log _logger;
    public final List<Task> EMPTY_TASK_LIST = Collections.emptyList();

    private EventManager _eventManager;
    private TaskCreator _taskCreator;
    private TaskDecorator _taskDecorator;

    public void setTaskCreator(final TaskCreator taskCreator)
    {
        _taskCreator = taskCreator;
    }

    public void setTaskDecorator(final TaskDecorator taskDecorator)
    {
        _taskDecorator = taskDecorator;
    }

    public void setEventManager(final EventManager eventManager)
    {
        _eventManager = eventManager;
    }

    public void setLog(final Log logger)
    {
        _logger = logger;
    }

    public void setRandomizer(final Randomizer randomizer)
    {
        _randomizer = randomizer;
    }

    public synchronized void addTask(final Task task)
    {
        checkInitialized();
        if (task == null)
        {
            throw new NullPointerException("task");
        }

        // Move the task from removed to active if the task has been removed previously
        if (_removedTasks.contains(task))
        {
            _removedTasks.remove(task);
        }

        _tasks.add(task);
        _logger.info("Task '" + task.getName() + "' added");

        // Signal the fact
        _eventManager.publish(new TasksAvailableEvent());
    }

    public Task addTaskClass(final Class taskClassName, final Task... followers)
    {
        final Task task = _taskCreator.createTask(taskClassName);
        task.setFollowers(Arrays.asList(followers));
        addTask(task);
        return task;
    }

    public Task addTaskInstance(final Object taskInstance, final Task... followers)
    {
        final Task task = _taskDecorator.decorateInstance(taskInstance);
        task.setFollowers(Arrays.asList(followers));
        addTask(task);
        return task;
    }

    public Task getNextRandomTask( )
    {
        checkInitialized();
        if (_availableTasks.size() == 0)
        {
            _logger.debug("No active tasks available");
            return null;
        }
        final int nextRandom = _randomizer.getRandomInt(_availableTasks.size());
        return _availableTasks.get(nextRandom);
    }

    public List<Task> getTasks()
    {
        checkInitialized();
        return Collections.unmodifiableList(_availableTasks);
    }

    public List<Task> getRegisteredTasks()
    {
        checkInitialized();
        return Collections.unmodifiableList(_tasks);
    }

    public Task getTaskByName(final String name)
    {
        checkInitialized();
        for (final Task task : _tasks)
        {
            if (task.getName().equals(name))
            {
                return task;
            }
        }
        for (final Task removedTask : _removedTasks)
        {
            if (removedTask.getName().equals(name))
            {
                return removedTask;
            }
        }
        return null;
    }

    public void removeTask(final Task task)
    {
        checkInitialized();
        if (_availableTasks.contains(task))
        {
            _logger.info("Removing task '" + task.getName() + "' from active list");
            _availableTasks.remove(task);
            _removedTasks.add(task);

            // Have we run out of tasks?
            if (_tasks.size() == 0)
            {
                _eventManager.publish(new NoTasksAvailableEvent());
            }

        }
        else
        {
            _logger.warn("Task '" + task.getName() + "' not in active list, can not remove");
        }

    }

    public List<Task> getRemovedTasks()
    {
        checkInitialized();
        return Collections.unmodifiableList(_removedTasks);
    }

    private void checkInitialized()
    {
        if (_tasks == null)
        {
            throw new IllegalStateException("not initialized");
        }
    }

    public int getNumberOfRemovedTasks()
    {
        checkInitialized();
        return _removedTasks.size();
    }

    public int getNumberOfTasks()
    {
        checkInitialized();
        return _availableTasks.size();
    }

    public void setTaskInfos(final List<TaskInfo> taskInfos)
    {
        if (taskInfos == null)
        {
            throw new NullPointerException("taskInfos");
        }
        _taskInfos = taskInfos;
    }

    public void initializeService()
    {
        if (_randomizer == null)
        {
            throw new IllegalStateException("no randomizer");
        }

        if (_logger == null)
        {
            throw new IllegalStateException("no log");
        }

        _removedTasks = new ArrayList<Task>();
        _tasks = new ArrayList<Task>();
        _availableTasks = new ArrayList<Task>();

        if (_taskInfos != null)
        {

            if (_taskInfos.size() == 0)
            {
                _logger.warn("No TaskInfo instances supplied");
                return;
            }

            for (final TaskInfo taskInfo : _taskInfos)
            {
                final Task task = taskInfo.getTask();
                // Pull in the possible followers ...
                if (taskInfo.getFollowers() != null && taskInfo.getFollowers().size() > 0)
                {
                    task.setFollowers(taskInfo.getFollowers());
                }
                else
                {
                    task.setFollowers(EMPTY_TASK_LIST);
                }
                _tasks.add(task);
                _logger.debug("Task '" + task.getName() + "' added - " + task.getFollowers().size() + " follower(s)");
            }
            _logger.info("'" + _tasks.size() + "' task(s) added");
        }
        else
        {
            _logger.warn("No TaskInfo instances supplied");
        }
    }

    public void basherEvent(final BasherEvent basherEvent)
    {
        if (basherEvent instanceof PhaseTransitionEvent)
        {
            PhaseTransitionEvent phaseTransitionEvent = (PhaseTransitionEvent) basherEvent;
            Phase newPhase = phaseTransitionEvent.getNewPhase();
            sortAvailableTasks(newPhase);
        }
    }

    void sortAvailableTasks(final Phase newPhase)
    {
        // Clear existing lists
        _removedTasks.clear();
        _availableTasks.clear();

        for (Task task : _tasks)
        {
            if (task.applicablePhases().contains(newPhase))
            {
                _availableTasks.add(task);
            }
        }

    }
}
