/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.impl;

import junit.framework.TestCase;
import net.sourceforge.basher.internal.Randomizer;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestInMemoryEntityRegistry extends TestCase
{

    public void testRegister()
    {
        InMemoryEntityRegistry inMemoryEntityRegistry = new InMemoryEntityRegistry();

        MockControl randomizerControl = MockControl.createControl(Randomizer.class);
        Randomizer randomizer = (Randomizer) randomizerControl.getMock();
        inMemoryEntityRegistry.setRandomizer(randomizer);

        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);

        randomizerControl.replay();


        final String entity = "SOME OBJECT";
        final String entityIdentifier = "AN ID";
        inMemoryEntityRegistry.register(entityIdentifier, entity);

        Object regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier);
        assertNotNull("Unexpected null returned from entity registry", regEntity);
        assertEquals("Invalud entity returned", entity, regEntity);

        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getNumRegistered(entityIdentifier));
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getAllRegistered(entityIdentifier).size());
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 10).size());


        randomizerControl.verify();


    }

    public void testRegisterManyOfSameType()
    {
        InMemoryEntityRegistry inMemoryEntityRegistry = new InMemoryEntityRegistry();

        MockControl randomizerControl = MockControl.createControl(Randomizer.class);
        Randomizer randomizer = (Randomizer) randomizerControl.getMock();
        inMemoryEntityRegistry.setRandomizer(randomizer);

        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(1);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(3);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(4);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(1);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(2);

        randomizerControl.replay();


        final String entity1 = "SOME OBJECT";
        final String entity2 = "SOME OBJECT";
        final String entity3 = "SOME OBJECT";
        final String entity4 = "SOME OBJECT";
        final String entity5 = "SOME OBJECT";
        final String entityIdentifier = "AN ID";
        inMemoryEntityRegistry.register(entityIdentifier, entity1);
        inMemoryEntityRegistry.register(entityIdentifier, entity2);
        inMemoryEntityRegistry.register(entityIdentifier, entity3);
        inMemoryEntityRegistry.register(entityIdentifier, entity4);
        inMemoryEntityRegistry.register(entityIdentifier, entity5);

        Object regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier);
        assertNotNull("Unexpected null returned from entity registry", regEntity);

        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getNumRegistered(entityIdentifier));
        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getAllRegistered(entityIdentifier).size());
        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 10).size());
        assertEquals("Wrong number of entities returned", 2, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 2).size());

        randomizerControl.verify();

    }


    public void testRegisterDifferentTypes()
    {
        InMemoryEntityRegistry inMemoryEntityRegistry = new InMemoryEntityRegistry();

        MockControl randomizerControl = MockControl.createControl(Randomizer.class);
        Randomizer randomizer = (Randomizer) randomizerControl.getMock();
        inMemoryEntityRegistry.setRandomizer(randomizer);


        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);

        randomizerControl.replay();

        final String entity = "SOME OBJECT";
        final String entityIdentifier = "AN ID";
        inMemoryEntityRegistry.register(entityIdentifier, entity);

        final String entity2 = "SOME DIFFERENT OBJECT";
        final String entityIdentifier2 = "A DIFFERENT ID";
        inMemoryEntityRegistry.register(entityIdentifier2, entity2);


        Object regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier);
        assertNotNull("Unexpected null returned from entity registry", regEntity);
        assertEquals("Invalud entity returned", entity, regEntity);
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getNumRegistered(entityIdentifier));
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getAllRegistered(entityIdentifier).size());
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 10).size());

        regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier2);
        assertNotNull("Unexpected null returned from entity registry", regEntity);
        assertEquals("Invalud entity returned", entity2, regEntity);
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getNumRegistered(entityIdentifier2));
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getAllRegistered(entityIdentifier2).size());
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getRandomSelection(entityIdentifier2, 10).size());

        randomizerControl.verify();
    }

    public void testUnregister()
    {
        InMemoryEntityRegistry inMemoryEntityRegistry = new InMemoryEntityRegistry();

        MockControl randomizerControl = MockControl.createControl(Randomizer.class);
        Randomizer randomizer = (Randomizer) randomizerControl.getMock();
        inMemoryEntityRegistry.setRandomizer(randomizer);

        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(1);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(3);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(4);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(4);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(4);
        randomizer.getRandomInt(5);
        randomizerControl.setReturnValue(4);

        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(4);
        randomizerControl.setReturnValue(2);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);
        randomizer.getRandomInt(1);
        randomizerControl.setReturnValue(0);

        randomizerControl.replay();

        final String entity1 = "SOME OBJECT";
        final String entity2 = "SOME OBJECT";
        final String entity3 = "SOME OBJECT";
        final String entity4 = "SOME OBJECT";
        final String entity5 = "SOME OBJECT";
        final String entityIdentifier = "AN ID";
        inMemoryEntityRegistry.register(entityIdentifier, entity1);
        inMemoryEntityRegistry.register(entityIdentifier, entity2);
        inMemoryEntityRegistry.register(entityIdentifier, entity3);
        inMemoryEntityRegistry.register(entityIdentifier, entity4);
        inMemoryEntityRegistry.register(entityIdentifier, entity5);

        final String entity6 = "SOME DIFFERENT OBJECT";
        final String entityIdentifier6 = "A DIFFERENT ID";
        inMemoryEntityRegistry.register(entityIdentifier6, entity6);


        Object regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier);
        assertNotNull("Unexpected null returned from entity registry", regEntity);

        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getNumRegistered(entityIdentifier));
        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getAllRegistered(entityIdentifier).size());
        assertEquals("Wrong number of entities registered", 5, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 10).size());
        assertEquals("Wrong number of entities returned", 2, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 2).size());

        inMemoryEntityRegistry.unregister(entityIdentifier, entity1);

        regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier);
        assertNotNull("Unexpected null returned from entity registry", regEntity);

        assertEquals("Wrong number of entities registered", 4, inMemoryEntityRegistry.getNumRegistered(entityIdentifier));
        assertEquals("Wrong number of entities registered", 4, inMemoryEntityRegistry.getAllRegistered(entityIdentifier).size());
        assertEquals("Wrong number of entities registered", 4, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 10).size());
        assertEquals("Wrong number of entities returned", 4, inMemoryEntityRegistry.getRandomSelection(entityIdentifier, 5).size());

        regEntity = inMemoryEntityRegistry.getRandom(entityIdentifier6);
        assertNotNull("Unexpected null returned from entity registry", regEntity);
        assertEquals("Invalud entity returned", entity6, regEntity);
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getNumRegistered(entityIdentifier6));
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getAllRegistered(entityIdentifier6).size());
        assertEquals("Wrong number of entities registered", 1, inMemoryEntityRegistry.getRandomSelection(entityIdentifier6, 10).size());

        randomizerControl.verify();

    }


}
