/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import junit.framework.TestCase;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.events.EventManager;
import net.sourceforge.basher.events.PhaseTransitionEvent;
import net.sourceforge.basher.internal.Randomizer;
import net.sourceforge.basher.tasks.ATestTask;
import org.apache.commons.logging.Log;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestTaskManagerImpl extends TestCase
{
    private MockControl _logControl;
    private MockControl _randomizerControl;
    private MockControl _eventManagerControl;

    private Randomizer _randomizer;
    private Log _log;
    private EventManager _eventManager;

    public void testInitilizationEmptyTaskInfoList()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        taskManagerImpl.setTaskInfos(Collections.EMPTY_LIST);

        _log.warn("No TaskInfo instances supplied");

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();
        verify();


    }

    public void testInitilizationNotWired()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();
        initializeControls();
        setReplay(_logControl, _randomizerControl);
        try
        {
            taskManagerImpl.initializeService();
        }
        catch (IllegalStateException e)
        {
            assertEquals("no randomizer", e.getMessage());
        }

        try
        {
            taskManagerImpl.setRandomizer(_randomizer);
            taskManagerImpl.initializeService();
        }
        catch (IllegalStateException e)
        {
            assertEquals("no log", e.getMessage());
        }
        verify();
    }

    public void testNotInitialized()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();
        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);
        setReplay(_logControl, _randomizerControl);

        try
        {
            taskManagerImpl.addTask(null);
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.getNextRandomTask();
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.getNumberOfRemovedTasks();
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.getNumberOfTasks();
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.getTaskByName("fred");
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.getTasks();
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }
        try
        {
            taskManagerImpl.removeTask(null);
        }
        catch (IllegalStateException e)
        {
            assertEquals("not initialized", e.getMessage());
        }

        verify();
    }


    public void testNullAddTask()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);
        _log.warn("No TaskInfo instances supplied");
        setReplay(_logControl, _randomizerControl);
        taskManagerImpl.initializeService();

        try
        {
            taskManagerImpl.addTask(null);
        }
        catch (NullPointerException e)
        {
            // should happne
            assertEquals("Bad message", "task", e.getMessage());
        }

    }

    public void testNullSetTaskInfo()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        try
        {
            taskManagerImpl.setTaskInfos(null);
        }
        catch (NullPointerException e)
        {
            // should happne
            assertEquals("Bad message", "taskInfos", e.getMessage());
        }

    }

    public void testInitilizationTwoItemTaskInfoList()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        List taskInfos = new ArrayList();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);
        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 0 follower(s)");

        _log.info("'2' task(s) added");

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        final List<Task> tasks = taskManagerImpl.getRegisteredTasks();

        assertEquals("Wrong number of tasks in list", 2, tasks.size());

        assertEquals("Invalid task in list", task1, tasks.get(0));
        assertEquals("Invalid task in list", task2, tasks.get(1));

        final Task task1Retrieved = taskManagerImpl.getTaskByName("task1");

        assertEquals("Wrong task retrieved by name", task1, task1Retrieved);

        final Task task2Retrieved = taskManagerImpl.getTaskByName("task2");
        assertEquals("Wrong task retrieved by name", task2, task2Retrieved);
        verify();

    }

    public void testInitilizationTwoItemTaskInfoListWithFollower()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        List taskInfos = new ArrayList();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);
        final ATestTask task2Follower = new ATestTask("task2Follower");
        final ArrayList followers = new ArrayList();
        followers.add(task2Follower);
        taskInfo.setFollowers(followers);
        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 1 follower(s)");

        _log.info("'2' task(s) added");

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        final List<Task> tasks = taskManagerImpl.getRegisteredTasks();

        assertEquals("Wrong number of tasks in list", 2, tasks.size());

        assertEquals("Invalid task in list", task1, tasks.get(0));
        assertEquals("Invalid task in list", task2, tasks.get(1));

        final Task task1Retrieved = taskManagerImpl.getTaskByName("task1");

        assertEquals("Wrong task retrieved by name", task1, task1Retrieved);

        final Task task2Retrieved = taskManagerImpl.getTaskByName("task2");
        assertEquals("Wrong task retrieved by name", task2, task2Retrieved);

        assertEquals("Wrong number of followers", 0, task1Retrieved.getFollowers().size());
        assertEquals("Wrong number of followers", 1, task2Retrieved.getFollowers().size());

        verify();

    }


    public void testGetRandomTask()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        List taskInfos = new ArrayList();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);
        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 0 follower(s)");

        _log.info("'2' task(s) added");


        _randomizer.getRandomInt(2);
        _randomizerControl.setReturnValue(1);

        _randomizer.getRandomInt(2);
        _randomizerControl.setReturnValue(0);

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        taskManagerImpl.basherEvent(new PhaseTransitionEvent(new BasherContext(), Phase.START, Phase.RUN));

        Task task2Retrieved = taskManagerImpl.getNextRandomTask();

        assertEquals("Wrong task retrived by random", task2, task2Retrieved);

        Task task1Retrieved = taskManagerImpl.getNextRandomTask();
        assertEquals("Wrong task retrived by random", task1, task1Retrieved);
        verify();


    }

    public void testRemoveTask()
    {
        final TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        final List<TaskInfo> taskInfos = new ArrayList<TaskInfo>();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);
        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 0 follower(s)");


        _log.info("'2' task(s) added");

        _log.info("Removing task 'task2' from active list");
        _log.warn("Task 'task2' not in active list, can not remove");

        _randomizer.getRandomInt(1);
        _randomizerControl.setReturnValue(0);

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        taskManagerImpl.sortAvailableTasks(Phase.RUN);

        taskManagerImpl.removeTask(task2);

        Task task1Retrieved = taskManagerImpl.getNextRandomTask();

        assertEquals("Wrong task retrived by random", task1, task1Retrieved);

        final List removedTasks = taskManagerImpl.getRemovedTasks();
        assertEquals("Wrong number of removed tasks", 1, removedTasks.size());

        assertEquals("Wrong number of removed tasks", 1, taskManagerImpl.getNumberOfRemovedTasks());

        final List tasks = taskManagerImpl.getTasks();
        assertEquals("Wrong number of active tasks", 1, tasks.size());
        assertEquals("Wrong number of active tasks", 1, taskManagerImpl.getNumberOfTasks());

        task1Retrieved = taskManagerImpl.getTaskByName("task1");
        assertEquals("Wrong task retrived by name", task1, task1Retrieved);

        taskManagerImpl.removeTask(task2);

        verify();

    }

    public void testAddRemovedTask()
    {
        final TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        final List<TaskInfo> taskInfos = new ArrayList<TaskInfo>();

        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);

        taskInfos.add(taskInfo);

        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);

        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 0 follower(s)");

        _log.info("'2' task(s) added");

        _log.info("Removing task 'task2' from active list");

        _log.info("Task 'task2' added");

        _randomizer.getRandomInt(1);
        _randomizerControl.setReturnValue(0);

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        taskManagerImpl.sortAvailableTasks(Phase.RUN);

        taskManagerImpl.removeTask(task2);

        Task task1Retrieved = taskManagerImpl.getNextRandomTask();

        assertEquals("Wrong task retrived by random", task1, task1Retrieved);

        final List<Task> removedTasks = taskManagerImpl.getRemovedTasks();
        assertEquals("Wrong number of removed tasks", 1, removedTasks.size());

        assertEquals("Wrong number of removed tasks", 1, taskManagerImpl.getNumberOfRemovedTasks());

        final List<Task> tasks = taskManagerImpl.getTasks();
        assertEquals("Wrong number of active tasks", 1, tasks.size());
        assertEquals("Wrong number of active tasks", 1, taskManagerImpl.getNumberOfTasks());

        task1Retrieved = taskManagerImpl.getTaskByName("task1");
        assertEquals("Wrong task retrived by name", task1, task1Retrieved);

        final Task task2Retrieved = taskManagerImpl.getTaskByName("task2");
        assertEquals("Wrong task retrived by name", task2, task2Retrieved);


        taskManagerImpl.addTask(task2);

        verify();

    }

    public void testGetByName()
    {
        final TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        final List<TaskInfo> taskInfos = new ArrayList<TaskInfo>();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();
        final ATestTask task2 = new ATestTask("task2");
        taskInfo.setTask(task2);
        taskInfos.add(taskInfo);

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");
        _log.debug("Task 'task2' added - 0 follower(s)");
        _log.info("'2' task(s) added");

        _log.info("Removing task 'task2' from active list");

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();
        taskManagerImpl.sortAvailableTasks(Phase.RUN);

        final Task task1Retrieved = taskManagerImpl.getTaskByName("task1");
        assertEquals("Wrong task retrived by name", task1, task1Retrieved);

        final Task task2Retrieved = taskManagerImpl.getTaskByName("task2");
        assertEquals("Wrong task retrived by name", task2, task2Retrieved);

        assertNull("could retrieve non-existant task by name", taskManagerImpl.getTaskByName("not-there"));


        taskManagerImpl.removeTask(task2);
        assertNull("could retrieve non-existant task by name", taskManagerImpl.getTaskByName("not-there"));

        verify();

    }


    public void testAddTask()
    {
        final TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer, _eventManager);

        List taskInfos = new ArrayList();
        TaskInfo taskInfo = new TaskInfo();
        final ATestTask task1 = new ATestTask("task1");
        taskInfo.setTask(task1);
        taskInfos.add(taskInfo);
        taskInfo = new TaskInfo();

        taskManagerImpl.setTaskInfos(taskInfos);

        _log.debug("Task 'task1' added - 0 follower(s)");

        _log.info("'1' task(s) added");
        _log.info("Task 'task2' added");

        _randomizer.getRandomInt(2);
        _randomizerControl.setReturnValue(1);

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        final ATestTask task2 = new ATestTask("task2");

        taskManagerImpl.addTask(task2);

        taskManagerImpl.sortAvailableTasks(Phase.RUN);


        Task task2Retrieved = taskManagerImpl.getNextRandomTask();

        assertEquals("Wrong task retrived by random", task2, task2Retrieved);

        List tasks = taskManagerImpl.getTasks();
        assertEquals("Wrong number of active tasks", 2, tasks.size());
        assertEquals("Wrong number of active tasks", 2, taskManagerImpl.getNumberOfTasks());

        task2Retrieved = taskManagerImpl.getTaskByName("task2");
        assertEquals("Wrong task retrived by name", task2, task2Retrieved);

        verify();

    }

    public void testGetRandomTaskEmptyActiveList()
    {
        TaskManagerImpl taskManagerImpl = new TaskManagerImpl();

        initializeControls();
        initializeTaskManager(taskManagerImpl, _log, _randomizer,_eventManager);

        _log.warn("No TaskInfo instances supplied");
        _log.debug("No active tasks available");

        setReplay(_logControl, _randomizerControl);

        taskManagerImpl.initializeService();

        Task taskRetrieved = taskManagerImpl.getNextRandomTask();

        assertNull("Unexpected task returned by random", taskRetrieved);
        verify();

    }


    private void initializeControls()
    {
        _logControl = MockControl.createControl(Log.class);
        _randomizerControl = MockControl.createControl(Randomizer.class);
        _eventManagerControl = MockControl.createControl(EventManager.class);

        _log = (Log) _logControl.getMock();
        _randomizer = (Randomizer) _randomizerControl.getMock();
        _eventManager = (EventManager) _eventManagerControl.getMock();

    }

    private void initializeTaskManager(final TaskManagerImpl taskManagerImpl, final Log log, final Randomizer randomizer, final EventManager eventManager)
    {
        taskManagerImpl.setLog(log);
        taskManagerImpl.setRandomizer(randomizer);
        taskManagerImpl.setEventManager(eventManager);
    }

    private void setReplay(final MockControl logControl, final MockControl randomizerControl)
    {
        logControl.replay();
        randomizerControl.replay();
    }

    private void verify()
    {
        _logControl.verify();
        _randomizerControl.verify();
    }


}
