/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.internal.impl;

import java.util.ArrayList;

import junit.framework.TestCase;
import net.sourceforge.basher.Collector;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskFailedException;
import net.sourceforge.basher.TaskManager;
import net.sourceforge.basher.internal.Randomizer;
import net.sourceforge.basher.internal.TimeSource;
import net.sourceforge.basher.tasks.ATestTask;
import org.apache.commons.logging.Log;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestTaskInvokerImpl extends TestCase
{
    private MockControl _collectorControl;
    private MockControl _logControl;
    private MockControl _randomizerControl;
    private MockControl _taskManagerControl;
    private MockControl _timeSourceControl;
    private Collector _collector;
    private Log _log;
    private Randomizer _randomizer;
    private TaskManager _taskManager;
    private TimeSource _timeSource;

    public void testNotInvokeWeight()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(50);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(10);

        _log.debug("Not running task: net.sourceforge.basher.tasks.ATestTask (weigth: 50 / weigher: 50)");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setWeight(50);

        _collector.notRun(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        verify();


    }

    public void testInvokeWeight()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(10);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);


        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();


        _collector.success(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        verify();


    }

    public void testNotInvokeBeforeStartTime()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(10);

        _log.debug("Not running task net.sourceforge.basher.tasks.ATestTask (not within time boundary 0/100/9223372036854775807)");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setRunFrom(100);

        _collector.notRun(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        verify();


    }

    public void testNotInvokeAfterEndTime()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(10000);

        _log.debug("Not running task net.sourceforge.basher.tasks.ATestTask (not within time boundary 10/0/1)");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setStopAfter(1);

        _collector.notRun(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);

        taskInvokerImpl.invokeTask(task);

        verify();

    }

    public void testInvokeAfterStartTime()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(11000);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);


        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setRunFrom(10);


        _collector.success(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);

        verify();

    }

    public void testInvokeBeforeEndTime()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);


        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setStopAfter(1);

        _collector.success(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        verify();


    }


    public void testInvokeLessThanMaxInvocations()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);
        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);

        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");
        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setMaxInvocations(2);

        _collector.success(task, 0);
        _collector.success(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        taskInvokerImpl.invokeTask(task);
        verify();


    }

    public void testInvokeUnlimitedMaxInvocations()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);
        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);


        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");
        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setMaxInvocations(0);

        _collector.success(task, 0);
        _collector.success(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        taskInvokerImpl.invokeTask(task);

        verify();

    }


    public void testNotInvokeLessThanMaxInvocations()
    {

        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);
        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);


        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");
        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");
        _log.debug("Not running task: net.sourceforge.basher.tasks.ATestTask (max invocations reached)");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        Task task = new ATestTask();
        task.setMaxInvocations(2);

        _collector.success(task, 0);
        _collector.success(task, 0);
        _collector.notRun(task, 0);

        _taskManager.removeTask(task);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);


        taskInvokerImpl.invokeTask(task);
        taskInvokerImpl.invokeTask(task);
        taskInvokerImpl.invokeTask(task);

        verify();

    }

    public void testNotRun()
    {
        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);

        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");
        _log.debug("Task not run: net.sourceforge.basher.tasks.ATestTask (forced not run)");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        ATestTask task = new ATestTask();
        task.setDontRun(true);

        _collector.notRun(task, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);

        taskInvokerImpl.invokeTask(task);

        verify();

    }

    public void testFailed()
    {
        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);

        _log.debug("Invoking task: net.sourceforge.basher.tasks.ATestTask");

        final TaskFailedException taskFailedException = new TaskFailedException("forced failure", ATestTask.EXCEPTION);
        _log.debug("Task failed: net.sourceforge.basher.tasks.ATestTask", taskFailedException);

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        ATestTask task = new ATestTask();
        task.setFailWithError(true);

        _collector.fail(task, 0, taskFailedException);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);

        taskInvokerImpl.invokeTask(task);

        verify();

    }


    public void testInvokeFollowers()
    {
        TaskInvokerImpl taskInvokerImpl = new TaskInvokerImpl();

        initializeControls();

        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);
        _randomizer.getRandomInt(100);
        _randomizerControl.setReturnValue(40);

        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getElapsedTime();
        _timeSourceControl.setReturnValue(1);

        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);
        _timeSource.getCurrentTime();
        _timeSourceControl.setReturnValue(1);

        _log.debug("Invoking task: task1");
        _log.debug("Invoking followers");
        _log.debug("Invoking task: follower");

        initializeTaskInvoker(taskInvokerImpl, _collector, _log, _randomizer, _taskManager, _timeSource);

        ATestTask task = new ATestTask("task1");
        ATestTask followingTask = new ATestTask("follower");

        final ArrayList followers = new ArrayList();
        followers.add(followingTask);
        task.setFollowers(followers);

        _collector.success(task, 0);
        _collector.success(followingTask, 0);

        setReplay(_collectorControl, _logControl, _randomizerControl, _taskManagerControl, _timeSourceControl);

        taskInvokerImpl.invokeTask(task);

        verify();

    }


    private void setReplay(final MockControl collectorControl, final MockControl logControl, final MockControl randomizerControl, final MockControl taskManagerControl, final MockControl timeSourceControl)
    {
        collectorControl.replay();
        logControl.replay();
        randomizerControl.replay();
        taskManagerControl.replay();
        timeSourceControl.replay();
    }

    private void initializeControls()
    {
        _collectorControl = MockControl.createControl(Collector.class);
        _logControl = MockControl.createControl(Log.class);
        _randomizerControl = MockControl.createControl(Randomizer.class);
        _taskManagerControl = MockControl.createControl(TaskManager.class);
        _timeSourceControl = MockControl.createControl(TimeSource.class);

        _collector = (Collector) _collectorControl.getMock();
        _log = (Log) _logControl.getMock();
        _randomizer = (Randomizer) _randomizerControl.getMock();
        _taskManager = (TaskManager) _taskManagerControl.getMock();
        _timeSource = (TimeSource) _timeSourceControl.getMock();
    }

    private void initializeTaskInvoker(final TaskInvokerImpl taskInvokerImpl, final Collector collector, final Log log, final Randomizer randomizer, final TaskManager taskManager, final TimeSource timeSource)
    {
        taskInvokerImpl.setCollector(collector);
        taskInvokerImpl.setLog(log);
        taskInvokerImpl.setRandomizer(randomizer);
        taskInvokerImpl.setTaskManager(taskManager);
        taskInvokerImpl.setTimeSource(timeSource);
    }

    private void verify()
    {
        _taskManagerControl.verify();
        _logControl.verify();
        _randomizerControl.verify();
    }


}
