/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.internal.tasks;

import junit.framework.TestCase;
import net.sourceforge.basher.Scheduler;
import org.apache.commons.logging.Log;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestThreadIncrementTask extends TestCase
{
    private MockControl _logControl;
    private MockControl _schedulerControl;

    private Log _log;
    private Scheduler _scheduler;


    public void testInvoke()
    {

        initializeControls();

        _log.info("Thread increment of '1' thread(s)");

        _log.debug("Incrementing thread count by '1'");

        _scheduler.addThreads(1);


        replay();

        ThreadIncrementTask threadIncrementTask = new ThreadIncrementTask();

        assertEquals("bad max time", 0, threadIncrementTask.getMaxTime());

        threadIncrementTask.setLog(_log);
        threadIncrementTask.setScheduler(_scheduler);

        threadIncrementTask.setThreadIncrementCount(1);

        try
        {
            threadIncrementTask.executeTask();
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }

        verify();

    }

    public void testNotInvoke()
    {

        initializeControls();

        replay();

        ThreadIncrementTask threadIncrementTask = new ThreadIncrementTask();

        threadIncrementTask.setLog(_log);
        threadIncrementTask.setScheduler(_scheduler);

        threadIncrementTask.setThreadIncrementCount(0);

        try
        {
            threadIncrementTask.executeTask();
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }

        verify();

    }


    public void testInvokeFailed()
    {

        initializeControls();

        _log.info("Thread increment of '1' thread(s)");
        _log.debug("Incrementing thread count by '1'");

        _scheduler.addThreads(1);
        final NullPointerException nullPointerException = new NullPointerException("null");
        _schedulerControl.setThrowable(nullPointerException);

        _log.error(nullPointerException.getMessage(), nullPointerException);


        replay();

        ThreadIncrementTask threadIncrementTask = new ThreadIncrementTask();

        threadIncrementTask.setLog(_log);
        threadIncrementTask.setScheduler(_scheduler);

        threadIncrementTask.setThreadIncrementCount(1);

        try
        {
            threadIncrementTask.executeTask();
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }

        verify();

    }


    private void replay()
    {
        _logControl.replay();
        _schedulerControl.replay();
    }

    private void verify()
    {
        _logControl.verify();
        _schedulerControl.verify();
    }

    private void initializeControls()
    {
        _schedulerControl = MockControl.createControl(Scheduler.class);
        _logControl = MockControl.createControl(Log.class);

        _log = (Log) _logControl.getMock();
        _scheduler = (Scheduler) _schedulerControl.getMock();

    }


}
