/**
 *
 * openutils-elfunctions (http://www.openmindlab.com/lab/products/elfunctions.html)
 * Copyright(C) 2008-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package net.sourceforge.openutils.elfunctions;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.FastDateFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Utility methods for date/time handling mapped to EL functions to allow usage in jsp pages.
 * @author fgiust
 * @version $Id: DateElUtils.java 2122 2010-03-09 20:02:04Z fgiust $
 */
public class DateElUtils
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(DateElUtils.class);

    /**
     * internal constant value to indicate no built-in format was found for the input pattern
     * @see {@link FastDateFormat}
     */
    private static final int NO_FORMAT = -1337;

    // constants to support built-in date and time formatting
    private static final String FORMAT_SHORT = "short";

    private static final String FORMAT_MEDIUM = "medium";

    private static final String FORMAT_LONG = "long";

    private static final String FORMAT_FULL = "full";

    /**
     * internal utility method to retrieve FastDateFormat format constants from input strings
     * @param pattern the pattern string
     * @return an integer corresponding to a format in FastDateFormat, or 0 if no match
     */
    private static int getFormat(String pattern)
    {
        int out = NO_FORMAT;
        if (StringUtils.isBlank(pattern))
        {
            // skip to return
        }
        else if (FORMAT_SHORT.equalsIgnoreCase(pattern))
        {
            out = FastDateFormat.SHORT;
        }
        else if (FORMAT_MEDIUM.equalsIgnoreCase(pattern))
        {
            out = FastDateFormat.MEDIUM;
        }
        else if (FORMAT_LONG.equalsIgnoreCase(pattern))
        {
            out = FastDateFormat.LONG;
        }
        else if (FORMAT_FULL.equalsIgnoreCase(pattern))
        {
            out = FastDateFormat.FULL;
        }
        return out;
    }

    /**
     * internal utility method to format dates attempting built-in styles first, and then resorting to
     * {@link FastDateFormat#getInstance(String)}
     * @param dateTime the date and time to format
     * @param pattern the format pattern, or one (or two) of the built-in styles
     * @param date true to format the date part with the built-in style
     * @param time false to format the time part with the built-in style
     * @param locale formatter locale, default locale if null
     * @return the formatted date, or the empty string if dateTime was null or pattern was blank
     */
    private static String format(Calendar dateTime, String pattern, boolean date, boolean time, Locale locale)
    {
        if (dateTime == null || StringUtils.isBlank(pattern))
        {
            return StringUtils.EMPTY;
        }

        FastDateFormat instance;
        int dateStyle = NO_FORMAT;
        int timeStyle = NO_FORMAT;
        if (date || time)
        {
            dateStyle = timeStyle = getFormat(pattern.trim());
            // if the standard pattern didn't work and we are in the date and time case, try to retrieve two different
            // styles with syntax "date_style;time_style"
            if (dateStyle == NO_FORMAT && date && time)
            {
                String[] split = pattern.split(";");
                if (split.length == 2)
                {
                    dateStyle = getFormat(split[0].trim());
                    timeStyle = getFormat(split[1].trim());
                }
            }
        }
        if (date && time && dateStyle != NO_FORMAT && timeStyle != NO_FORMAT)
        {
            instance = FastDateFormat.getDateTimeInstance(dateStyle, timeStyle, locale);
        }
        else if (date && dateStyle != NO_FORMAT)
        {
            instance = FastDateFormat.getDateInstance(dateStyle, locale);
        }
        else if (time && timeStyle != NO_FORMAT)
        {
            instance = FastDateFormat.getTimeInstance(timeStyle, locale);
        }
        else
        {
            instance = FastDateFormat.getInstance(pattern, locale);
        }
        return instance.format(dateTime);
    }

    /**
     * formats a date using the given pattern or built-in style
     * @param date the date to format
     * @param pattern a valid pattern, or one of "short", "medium", "long", "full" (case insensitive)
     * @param languageId lowercase two-letter ISO-639 code.
     * @return the formatted date
     */
    public static String formatDate(Calendar date, String pattern, String languageId)
    {
        return format(date, pattern, true, false, new Locale(languageId));
    }

    /**
     * formats a time using the given pattern or built-in style
     * @param time the time to format
     * @param pattern a valid pattern, or one of "short", "medium", "long", "full" (case insensitive)
     * @param languageId lowercase two-letter ISO-639 code.
     * @return the formatted time
     */
    public static String formatTime(Calendar time, String pattern, String languageId)
    {
        return format(time, pattern, false, true, new Locale(languageId));
    }

    /**
     * Formats a date and/or time using the given pattern or built-in style - supports composite formats for date and
     * time with the syntax "date_format;time_format" (ex: "long;short")
     * @param dateTime the date and time
     * @param pattern date pattern
     * @param languageId lowercase two-letter ISO-639 code.
     * @return formatted string
     */
    public static String formatDateTime(Calendar dateTime, String pattern, String languageId)
    {
        return format(dateTime, pattern, true, true, new Locale(languageId));
    }

    /**
     * Formats a date and/or time using the given pattern or built-in style - supports composite formats for date and
     * time with the syntax "date_format;time_format" (ex: "long;short") with locale
     * @param dateTime the date and time
     * @param languageId lowercase two-letter ISO-639 code.
     * @param pattern date pattern
     * @return formatted string
     */
    public static String formatDateTimeWithLocale(Calendar dateTime, String languageId, String pattern)
    {
        return format(dateTime, pattern, true, true, new Locale(languageId));
    }

    /**
     * format a milliseconds interval as a string like 1y 2d 23h 22m 18s (year estimation approximates to 365 days)
     * @param millisInterval the interval in milliseconds
     * @return the formatted string
     */
    public static String formatInterval(Long millisInterval)
    {

        if (millisInterval == null)
        {
            return StringUtils.EMPTY;
        }
        long secondsInterval = millisInterval / 1000;
        long years = secondsInterval / (365 * 24 * 60 * 60);
        long days = (secondsInterval % (365 * 24 * 60 * 60)) / (24 * 60 * 60);
        long hours = (secondsInterval % (24 * 60 * 60)) / (60 * 60);
        long minutes = (secondsInterval % (60 * 60)) / 60;
        long seconds = secondsInterval % 60;
        StringBuilder out = new StringBuilder();

        if (years > 0)
        {
            out.append(years).append("y ");
        }
        if (days > 0)
        {
            out.append(days).append("d ");
        }
        if (hours > 0)
        {
            out.append(hours).append("h ");
        }
        if (minutes > 0)
        {
            out.append(minutes).append("m ");
        }
        out.append(seconds).append('s');

        return out.toString();
    }

    /**
     * Retrieve the milliseconds in difference between now and the input date
     * @param date the date
     * @return positive value for dates older than now
     */
    public static Long getMillisFromNow(Calendar date)
    {
        if (date == null)
        {
            return 0L;
        }
        return System.currentTimeMillis() - date.getTimeInMillis();
    }

    /**
     * parse a date with a given pattern and return the parsed date as a calendar object (null safe)
     * @param date the date string
     * @param pattern the pattern
     * @return the parsed date as a Calendar, or null
     */
    public static Calendar parseDate(String date, String pattern)
    {
        if (StringUtils.isNotEmpty(date) && StringUtils.isNotEmpty(pattern))
        {
            Calendar cal = Calendar.getInstance();
            try
            {
                cal.setTime(new SimpleDateFormat(pattern).parse(date));
                return cal;
            }
            catch (ParseException e)
            {
                log.warn("unable to parse date from string: " + date + " with pattern: " + pattern, e);
            }
        }
        return null;
    }

    /**
     * parse a date with a XSD pattern (YYYY-MM-DDThh:mm:ss) and return the parsed date as as a calendar object
     * @param xsdDate the date string in the format "yyyy-MM-ddThh:mm:ss"
     * @return the parsed date as a Calendar
     */
    public static Calendar parseXsdDate(String xsdDate)
    {

        if (StringUtils.isEmpty(xsdDate))
        {
            return null;
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");
        Calendar cal = Calendar.getInstance();
        try
        {
            cal.setTime(sdf.parse(xsdDate));
            return cal;
        }
        catch (ParseException e)
        {
            log.warn("Unable to parse date {}", xsdDate);
        }

        return null;
    }

    /**
     * Get a date in the XSD format "yyyy-MM-ddThh:mm:ss:+01:00"
     * @param date input calendar
     * @return XSD formatted date
     */
    public static String toXsdDate(Calendar date)
    {
        if (date == null)
        {
            return null;
        }

        FastDateFormat sdf = FastDateFormat.getInstance("yyyy-MM-dd'T'hh:mm:ssZ");

        String xsdDate = sdf.format(date);
        int length = xsdDate.length();

        return StringUtils.substring(xsdDate, 0, length - 2) + ":" + StringUtils.substring(xsdDate, length - 2, length);

    }

    /**
     * Given a Calendar, returns a new calendar with same day and hh:mm:ss:mm set to 00:00:00:00
     * @param calendar the calendar
     * @return a new calendar with same day and hh:mm:ss:mm set to 00:00:00:00
     */
    public static Calendar getMidnight00(Calendar calendar)
    {
        Calendar midniteCalendar = (Calendar) calendar.clone();
        midniteCalendar.set(Calendar.HOUR_OF_DAY, 0);
        midniteCalendar.set(Calendar.MINUTE, 0);
        midniteCalendar.set(Calendar.SECOND, 0);
        midniteCalendar.set(Calendar.MILLISECOND, 0);
        return midniteCalendar;
    }

    /**
     * Given a Calendar, returns a new calendar with day set to day after and hh:mm:ss:mm set to 00:00:00:00
     * @param calendar The date to get yymmdd from
     * @return a new calendar with day set to day after and hh:mm:ss:mm set to 00:00:00:00
     */
    public static Calendar getMidnight24(Calendar calendar)
    {

        Calendar midniteCalendar = (Calendar) calendar.clone();
        midniteCalendar.add(Calendar.DAY_OF_MONTH, 1);
        midniteCalendar.set(Calendar.HOUR_OF_DAY, 0);
        midniteCalendar.set(Calendar.MINUTE, 0);
        midniteCalendar.set(Calendar.SECOND, 0);
        midniteCalendar.set(Calendar.MILLISECOND, 0);
        return midniteCalendar;
    }

    /**
     * Tests if the current date is between the specified time interval (limits included)
     * @param from interval start
     * @param to interval end
     * @return true if the current date falls within the specified interval, false otherwise
     */
    public static boolean currentDateIncluded(Calendar from, Calendar to)
    {
        if (from == null && to == null)
        {
            return true;
        }

        Calendar now = Calendar.getInstance();

        if (from != null && now.before(from))
        {
            return false;
        }
        if (to != null && now.after(to))
        {
            return false;
        }
        return true;

    }

}
