/**
 *
 * openutils-elfunctions (http://www.openmindlab.com/lab/products/elfunctions.html)
 * Copyright(C) 2008-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package net.sourceforge.openutils.elfunctions;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.SystemUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.commons.lang.math.NumberUtils;


/**
 * Utility methods mapped to EL functions to allow usage in jsp pages.
 * @author fgiust
 * @version $Id: ElStringUtils.java 1696 2010-01-20 16:58:56Z fgiust $
 */
public class ElStringUtils
{

    /**
     * The space character.
     * @return the space character
     */
    public static String space()
    {
        return " ";
    }

    /**
     * The tab character.
     * @return the tab character
     */
    public static String tab()
    {
        return "\t";
    }

    /**
     * The newline (<code>\n</code>) character.
     * @return the newline (<code>\n</code>) character
     */
    public static String newline()
    {
        return "\n";
    }

    /**
     * Escapes a javascript string. If "true" is passed as parameter, the string is between ", if false is between '
     * @param text the string to be escaped
     * @param dbl If "true" the escaped string is returned between ", if false is returned between '
     * @return the escaped string between either ' or "
     */
    public static String escapeJsText(String text, boolean dbl)
    {
        String repl = dbl ? "\"" : "'";
        String with = "\\" + repl;
        return repl + (text != null ? StringUtils.replace(text, repl, with) : "") + repl;
    }

    /**
     * Crops a String to a given length, adding a suffix if needed.
     * @param value The string to be adapted
     * @param maxLength The number of chars of the string to be kept
     * @param ellipses The suffix to be added if the string is not complete
     * @return adapted String
     */
    public static String adaptStringLength(String value, int maxLength, String ellipses)
    {
        if (value != null && value.length() > maxLength)
        {
            return value.substring(0, maxLength) + ellipses;
        }

        return value;
    }

    /**
     * Strip any html tag from a String.
     * @param string string The string to be stripped
     * @return stripped String
     */
    public static String stripHtmlTags(String string)
    {
        if (StringUtils.isNotBlank(string))
        {
            return StringUtils.remove(StringUtils.remove(string.replaceAll("<(.|\n)+?>", ""), '\r'), '\n').trim();

        }
        return null;
    }

    /**
     * Splits the given strings on newlines (<code>\n</code>)
     * @param string string to be split
     * @return split string
     */
    public static String[] splitNewlines(String string)
    {
        return StringUtils.splitPreserveAllTokens(StringUtils.replace(string, "\r", ""), '\n');
    }

    /**
     * Splits the given strings on tabs (<code>\t</code>)
     * @param string string to be split
     * @return split string
     */
    public static String[] splitOnTabs(String string)
    {
        return StringUtils.splitPreserveAllTokens(string, '\t');
    }

    /**
     * Tests if this string ends with the specified suffix.
     * @param string string to evaluate
     * @param suffix suffix to be added
     * @return true if the string ends with the suffix, false otherwise
     */
    public static boolean endsWith(String string, String suffix)
    {
        return string.endsWith(suffix);
    }

    /**
     * Parse a number, passed as String, and return his Long
     * @param value the string to parse
     * @return the parsed long, or <code>0L</code> if parse fails
     * @see {@link NumberUtils#toLong()}
     */
    public static Long toLong(String value)
    {
        return NumberUtils.toLong(value, 0);
    }

    /**
     * Parse a number, passed as String, and return his closest rounding as a long (or 0L if parse fails)
     * @param value the string to parse
     * @return the closest long to the parsed double, or <code>0L</code> if parse fails
     * @see {@link Math#round(double)}
     */
    public static Long toRoundedLong(String value)
    {
        double doubleValue = NumberUtils.toDouble(value);
        if (doubleValue != 0.0d)
        {
            return Math.round(doubleValue);
        }
        return 0L;
    }

    /**
     * Parse a double, passed as String, and return his floor as a long (or 0L if parse fails)
     * @param value the string to parse
     * @return the parsed double's floor as a long, or <code>0L</code> if parse fails
     * @see {@link Math#floor(double)}
     */
    public static Long toFloorLong(String value)
    {
        double doubleValue = NumberUtils.toDouble(value);
        if (doubleValue != 0.0d)
        {
            return Math.round(Math.floor(doubleValue));
        }
        return 0L;
    }

    /**
     * Parse a double, passed as String, and return his ceil as a long (or 0L if parse fails)
     * @param value the string to parse
     * @return the parsed double's floor as a long, or <code>0L</code> if parse fails
     * @see {@link Math#ceil(double)}
     */
    public static Long toCeilLong(String value)
    {
        double doubleValue = NumberUtils.toDouble(value);
        if (doubleValue != 0.0d)
        {
            return Math.round(Math.ceil(doubleValue));
        }
        return 0L;
    }

    /**
     * Shorten a text with a number of lines and a number of chars per line to be displayed. Display optional ellipses
     * where the line is shortened.
     * @param text original text
     * @param lines number of lines
     * @param numOfCharsPerLine number of chars per line
     * @param ellipses optional ellipses ('...') to display optional ellipses where the line is shortened.
     * @return cropped string
     */
    public static String shorten(String text, int lines, int numOfCharsPerLine, String ellipses)
    {
        if (text == null)
        {
            return null;
        }

        String wrapped = WordUtils.wrap(text, numOfCharsPerLine, SystemUtils.LINE_SEPARATOR, true);
        int j = 0;
        int lineCount = 0;
        while (lineCount < lines && j != -1)
        {
            j = StringUtils.indexOf(wrapped, SystemUtils.LINE_SEPARATOR, j + 1);
            lineCount++;
        }

        return (j != -1)
            ? StringUtils.substring(wrapped, 0, j) + (ellipses == null ? StringUtils.EMPTY : ellipses)
            : text;
    }

    /**
     * Encodes a url in UTF-8 format
     * @param string url to be encoded
     * @return a url UTF-8 encoded
     */
    public static String urlencode(String string)
    {
        try
        {
            return URLEncoder.encode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            // should never happen
            return string;
        }
    }

    /**
     * Decodes a url in UTF-8 format
     * @param string url to be decoded
     * @return a url UTF-8 decoded
     */
    public static String urldecode(String string)
    {
        try
        {
            return URLDecoder.decode(string, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            // should never happen
            return string;
        }
    }
}
