/*
 * Copyright (c) Openmind.  All rights reserved. http://www.openmindonline.it
 */
package it.openutils.hibernate.security.aop;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.ArrayList;
import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.GrantedAuthority;
import org.acegisecurity.context.SecurityContextHolder;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.hibernate.Filter;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: $
 */
@Aspect
public class AOPSecurity
{

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(AOPSecurity.class);

    private SecurityRuleManager securityRuleManager;

    private List<String> securedDAOs;

    private boolean enabled;

    /**
     * @param pjp The proceeding joinpoint
     * @param filter The entity we are going to filter
     * @param additionalCriteria The additional criteria list, cannot be null
     * @return The execution invocation result
     * @throws Throwable Any exception occurring in the invoked method
     */
    @Around("execution(* it.openutils.dao.hibernate.*.*(Object, .., java.util.List<org.hibernate.criterion.Criterion>)) && "
        + //
        " args(filter, .., additionalCriteria)")
    public Object applySecurityRules(ProceedingJoinPoint pjp, Object filter, List<Criterion> additionalCriteria)
        throws Throwable
    {
        if (!enabled)
        {
            log.debug("DAO security disabled, proceeding.");
            return pjp.proceed();
        }

        if (!securedDAOs.contains(pjp.getTarget().getClass().getCanonicalName()))
        {
            log.debug("The intercepted DAO {} is not secured, proceeding.", pjp.getTarget().toString());
            return pjp.proceed();
        }

        log.debug("applying security rules for {} with criteria {}", filter.toString(), additionalCriteria);
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null)
        {
            throw new SecurityException("Authentication is not valid");
        }
        GrantedAuthority[] authorities = authentication.getAuthorities();
        List<String> roles = new ArrayList<String>();
        for (int i = 0; i < authorities.length; i++)
        {
            roles.add(authorities[i].getAuthority());
        }

        String entity = filter.getClass().getCanonicalName();
        List<SecurityRule> rules = securityRuleManager.getRulesForRoles(entity, roles);

        if (rules.isEmpty())
        {
            String grantedRoles = StringUtils.EMPTY;
            for (int i = 0; i < authorities.length; i++)
            {
                grantedRoles += authorities[i].getAuthority() + " ";
            }
            log.warn("Access is denied on " + entity + ", for user {} with roles {}", SecurityContextHolder
                .getContext()
                .getAuthentication()
                .getPrincipal()
                .toString(), grantedRoles);

            throw new SecurityException("Access denied");
        }

        Filter hibernateFilter = securityRuleManager.getEntityFilterFromRules(entity, rules);

        Criterion sqlCriterion = Restrictions.sqlRestriction(hibernateFilter
            .getFilterDefinition()
            .getDefaultFilterCondition());

        log.debug("Adding sql restriction: {}", sqlCriterion.toString());
        additionalCriteria.add(sqlCriterion);

        Object result = pjp.proceed();
        return result;
    }

    /**
     * Sets the securityRuleManager.
     * @param securityRuleManager the securityRuleManager to set
     */
    public void setSecurityRuleManager(SecurityRuleManager securityRuleManager)
    {
        this.securityRuleManager = securityRuleManager;
    }

    /**
     * Sets the securedDAOs.
     * @param securedDAOs the securedDAOs to set
     */
    public void setSecuredDAOs(List<String> securedDAOs)
    {
        this.securedDAOs = securedDAOs;
    }


    /**
     * Sets the enabled.
     * @param enabled the enabled to set
     */
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }
}
