/*
 * Copyright (c) Openmind.  All rights reserved. http://www.openmindonline.it
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.SecurityRule;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Filter;
import org.hibernate.HibernateException;
import org.hibernate.engine.FilterDefinition;


/**
 * @author fcarone
 * @version $Id: $
 */
public class JavaBeanFilter implements Filter
{

    private FilterDefinition filterDefinition;

    /**
     * @param bean The bean to set rules for
     * @param securityRules The list of {@link SecurityRule}s to apply.
     * @throws ClassNotFoundException If the bean class has not been found
     * @throws InstantiationException If the bean doesn't contain the no-arg constructor
     * @throws IllegalAccessException If the bean properties cannot be accessed
     * @throws SecurityException If the bean class cannot be accessed
     * @throws NoSuchFieldException If the property contained in the security rule refers to a bean non-existent field
     */
    @SuppressWarnings("unchecked")
    public JavaBeanFilter(String bean, List<SecurityRule> securityRules)
        throws ClassNotFoundException,
        InstantiationException,
        IllegalAccessException,
        SecurityException,
        NoSuchFieldException
    {
        Class< ? extends Object> beanClass = Class.forName(bean, true, this.getClass().getClassLoader());

        if (!beanClass.isAnnotationPresent(Entity.class))
        {
            throw new IllegalArgumentException();
        }
        if (!(beanClass.isAnnotationPresent(Table.class) || beanClass
            .isAnnotationPresent(org.hibernate.annotations.Table.class)))
        {
            throw new IllegalArgumentException();
        }

        Map<String, String> propertyColumnMap = new HashMap<String, String>();

        String filterDefCondition = StringUtils.EMPTY;
        String filterName = StringUtils.EMPTY;

        Map<String, List<SecurityRule>> roleRuleMap = new LinkedHashMap<String, List<SecurityRule>>();
        for (SecurityRule securityRule : securityRules)
        {
            if (!roleRuleMap.containsKey(securityRule.getRole()))
            {
                roleRuleMap.put(securityRule.getRole(), new ArrayList<SecurityRule>());
            }
            roleRuleMap.get(securityRule.getRole()).add(securityRule);
        }

        for (Map.Entry<String, List<SecurityRule>> entry : roleRuleMap.entrySet())
        {
            filterName += entry.getKey();

            List<SecurityRule> rules = entry.getValue();
            if (!StringUtils.isEmpty(filterDefCondition))
            {
                filterDefCondition += " OR ";
            }
            filterDefCondition += "(";
            String subFilterCond = StringUtils.EMPTY;
            for (SecurityRule securityRule : rules)
            {
                String property = securityRule.getProperty();
                filterName += property;

                Field field = beanClass.getDeclaredField(property);

                // @todo: annotations may also be defined on getters/setters...
                propertyColumnMap.put(property, field.getAnnotation(Column.class).name());
                if (!StringUtils.isEmpty(subFilterCond))
                {
                    subFilterCond += " AND ";
                }
                String modifier = null;
                String startQuote = null;
                String endQuote = null;
                switch (securityRule.getModifier())
                {
                    case EQUALS :
                        modifier = " = ";
                        startQuote = "\'";
                        endQuote = startQuote;
                        break;

                    case NOT :
                        modifier = " != ";
                        startQuote = "\'";
                        endQuote = startQuote;
                        break;

                    default :
                        throw new IllegalArgumentException("Modifier " + securityRule.getModifier() + "not recognized");
                }
                subFilterCond += field.getAnnotation(Column.class).name()
                    + modifier
                    + startQuote
                    + securityRule.getValue()
                    + endQuote;

                filterName += securityRule.getValue();
            }
            filterDefCondition += subFilterCond;
            filterDefCondition += ")";
        }

        // filtername is unique, but untraceable
        this.filterDefinition = new FilterDefinition(
            Integer.toString(filterName.hashCode()),
            filterDefCondition,
            new HashMap());
    }

    /**
     * {@inheritDoc}
     */
    public FilterDefinition getFilterDefinition()
    {
        return this.filterDefinition;
    }

    /**
     * {@inheritDoc}
     */
    public String getName()
    {
        return this.filterDefinition.getFilterName();
    }

    /**
     * {@inheritDoc}
     */
    public Filter setParameter(String name, Object value)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public Filter setParameterList(String name, Collection values)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    public Filter setParameterList(String name, Object[] values)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    public void validate() throws HibernateException
    {
        //
    }

}
