/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.aop;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.ArrayList;
import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.GrantedAuthority;
import org.acegisecurity.context.SecurityContextHolder;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.hibernate.Filter;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: AOPSecurity.java 705 2008-02-26 10:56:55Z fcarone $
 */
@Aspect
public class AOPSecurity
{

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(AOPSecurity.class);

    private boolean denyIfNoRulesFound = true;

    private SecurityRuleManager securityRuleManager;

    private List<String> securedDAOs;

    private boolean enabled;

    /**
     * @param pjp The proceeding joinpoint
     * @param filter The entity we are going to filter
     * @param additionalCriteria The additional criteria list, cannot be null
     * @return The execution invocation result
     * @throws Throwable Any exception occurring in the invoked method
     */
    @Around("execution(* it.openutils.dao.hibernate.*.*(Object, .., java.util.List<org.hibernate.criterion.Criterion>)) && "
        + //
        " args(filter, .., additionalCriteria)")
    public Object applySecurityRules(ProceedingJoinPoint pjp, Object filter, List<Criterion> additionalCriteria)
        throws Throwable
    {
        if (!enabled)
        {
            log.debug("DAO security disabled, proceeding.");
            return pjp.proceed();
        }

        if (!securedDAOs.contains(pjp.getTarget().getClass().getCanonicalName()))
        {
            log.debug("The intercepted DAO {} is not secured, proceeding.", pjp.getTarget().toString());
            return pjp.proceed();
        }

        log.debug("applying security rules for {} with criteria {}", filter.toString(), additionalCriteria);
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null)
        {
            throw new SecurityException("Authentication is not valid");
        }
        GrantedAuthority[] authorities = authentication.getAuthorities();
        List<String> roles = new ArrayList<String>();
        for (int i = 0; i < authorities.length; i++)
        {
            roles.add(authorities[i].getAuthority());
        }

        String entity = filter.getClass().getCanonicalName();
        List<SecurityRule> rules = securityRuleManager.getRulesForRoles(filter, roles);

        if (rules.isEmpty())
        {
            if (log.isWarnEnabled())
            {
                String grantedRoles = StringUtils.EMPTY;
                for (int i = 0; i < authorities.length; i++)
                {
                    grantedRoles += authorities[i].getAuthority() + " ";
                }
                log.warn(
                    "No rules found for " + entity + ", user {} with roles {}",
                    SecurityContextHolder.getContext().getAuthentication().getPrincipal().toString(),
                    grantedRoles);
            }
            if (denyIfNoRulesFound)
            {
                log.debug("denyIfNoRulesFound is true, denying access.");
                throw new SecurityException("Access denied");
            }
            else
            {
                log.debug("denyIfNoRulesFound is false, allowing access.");
                return pjp.proceed();
            }
        }

        Filter hibernateFilter = securityRuleManager.getEntityFilterFromRules(entity, rules);

        Criterion sqlCriterion = Restrictions.sqlRestriction(hibernateFilter
            .getFilterDefinition()
            .getDefaultFilterCondition());

        log.debug("Adding sql restriction: {}", sqlCriterion.toString());
        additionalCriteria.add(sqlCriterion);

        Object result = pjp.proceed();
        return result;
    }

    /**
     * Sets the securityRuleManager.
     * @param securityRuleManager the securityRuleManager to set
     */
    public void setSecurityRuleManager(SecurityRuleManager securityRuleManager)
    {
        this.securityRuleManager = securityRuleManager;
    }

    /**
     * Sets the securedDAOs.
     * @param securedDAOs the securedDAOs to set
     */
    public void setSecuredDAOs(List<String> securedDAOs)
    {
        this.securedDAOs = securedDAOs;
    }

    /**
     * Sets the enabled.
     * @param enabled the enabled to set
     */
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }


    /**
     * Sets the denyIfNoRulesFound.
     * @param denyIfNoRulesFound the denyIfNoRulesFound to set
     */
    public void setDenyIfNoRulesFound(boolean denyIfNoRulesFound)
    {
        this.denyIfNoRulesFound = denyIfNoRulesFound;
    }
}
