/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.services;

import it.openutils.hibernate.security.dataobject.SecurityRule;

import java.util.List;

import org.hibernate.Filter;


/**
 * @author fcarone
 * @version $Id: SecurityRuleManager.java 705 2008-02-26 10:56:55Z fcarone $
 */
public interface SecurityRuleManager
{

    /**
     * @return All the defined {@link SecurityRule}s.
     */
    List<SecurityRule> findAll();

    /**
     * @param filter The {@link SecurityRule} to use as search filter
     * @return The list of matching {@link SecurityRule}s.
     */
    List<SecurityRule> findFiltered(SecurityRule filter);

    /**
     * @param id The id of the {@link SecurityRule} to load
     * @return The {@link SecurityRule} with the given id
     */
    SecurityRule load(Long id);

    /**
     * @param id The id of the {@link SecurityRule} to load
     * @return The {@link SecurityRule} with the given id, or null if it is not found
     */
    SecurityRule loadIfAvailable(Long id);

    /**
     * @param securityRule The security rule to remove
     */
    void delete(SecurityRule securityRule);

    /**
     * @param securityRule The {@link SecurityRule} to save or update
     */
    void saveOrUpdate(SecurityRule securityRule);

    /**
     * @param securityRule The {@link SecurityRule} to save
     * @return The id of the saved {@link SecurityRule}
     */
    Long save(SecurityRule securityRule);

    /**
     * @param securityRule The {@link SecurityRule} to update
     */
    void update(SecurityRule securityRule);

    /**
     * @param entity The entity to apply rules for
     * @param rules The list of rules to apply
     * @return The Hibernate Filter responding to the given criteria
     * @throws SecurityException
     * @throws ClassNotFoundException
     * @throws InstantiationException
     * @throws IllegalAccessException
     * @throws NoSuchFieldException
     */
    public Filter getEntityFilterFromRules(String entity, List<SecurityRule> rules) throws SecurityException,
        ClassNotFoundException, InstantiationException, IllegalAccessException, NoSuchFieldException;

    /**
     * @param entity The entity name we want rules for
     * @param roles The list of roles
     * @return The list of security rules matching entity - roles
     */
    List<SecurityRule> getRulesForRoles(String entity, List<String> roles);

    /**
     * @param entity The entity we want rules for
     * @param roles The list of roles
     * @return The list of security rules matching entity - roles
     */
    List<SecurityRule> getRulesForRoles(Object entity, List<String> roles);

}
