/*
 * Copyright 2005 Fabrizio Giustina.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.testing.junit;

import java.io.InputStream;
import java.sql.SQLException;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.commons.lang.ClassUtils;
import org.apache.commons.lang.StringUtils;
import org.dbunit.database.DatabaseConnection;
import org.dbunit.database.DatabaseSequenceFilter;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.FilteredDataSet;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.excel.XlsDataSet;
import org.dbunit.dataset.filter.AbstractTableFilter;
import org.dbunit.dataset.filter.ITableFilter;
import org.dbunit.dataset.filter.SequenceTableFilter;
import org.dbunit.dataset.xml.XmlDataSet;
import org.dbunit.ext.mssql.InsertIdentityOperation;
import org.dbunit.operation.DatabaseOperation;
import org.hibernate.SessionFactory;
import org.springframework.orm.hibernate3.SessionHolder;
import org.springframework.transaction.support.TransactionSynchronizationManager;


/**
 * Base class for running DAO tests.
 * @author fgiust
 * @version $Revision $ ($Author $)
 */
public abstract class DbUnitTestCase extends SpringTestCase
{

    /**
     * Hibernate session factory.
     */
    private SessionFactory sessionFactory;

    private static final String BASETEST_DELETE = "/_BaseDAOTest-delete.xml";

    protected static IDataSet truncateDataSet;

    /**
     * Should use deferred close emulating the spring OpenSessionInView filter? Default is <code>true</code>
     * @return <code>true</code> if deferred close should be used
     */
    protected boolean mimicSessionFilter()
    {
        return true;
    }

    /**
     * Returns the table filter that will be used to exclude certain tables from sort/deletion. This may be overridden
     * by subclasses, and is needed when tables have circular references (not handled by dbunit DatabaseSequenceFilter)
     * @return an <code>ITableFilter</code>
     */
    protected ITableFilter getTableFilter()
    {
        return new AbstractTableFilter()
        {

            @Override
            public boolean isValidName(String tableName) throws DataSetException
            {
                // default excludes:
                // $ = oracle recycle bin tables
                // JBPM = jbpm tables, with circular references
                return !StringUtils.contains(tableName, "$") && !StringUtils.contains(tableName, "JBPM");
            }
        };
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    @Override
    protected void setUp() throws Exception
    {
        super.setUp();

        // insert values
        IDataSet dataSet = null;

        String datesetFileName = "/" + ClassUtils.getShortClassName(getClass()) + "-load.xml";
        InputStream testData = getClass().getResourceAsStream(datesetFileName);

        if (testData != null)
        {
            if (log.isDebugEnabled())
            {
                log.debug("loading dataset {}", datesetFileName);
            }

            dataSet = new XmlDataSet(testData);
        }
        else
        {
            // check for excel
            datesetFileName = "/" + ClassUtils.getShortClassName(getClass()) + "-load.xls";
            testData = getClass().getResourceAsStream(datesetFileName);

            if (testData != null)
            {
                if (log.isDebugEnabled())
                {
                    log.debug("loading dataset {}", datesetFileName);
                }

                dataSet = new XlsDataSet(testData);
            }
        }

        if (dataSet == null)
        {
            log.debug("No test data found with name [{}]", datesetFileName);
        }
        else
        {

            IDatabaseConnection connection = createConnection();

            // truncate common tables
            if (truncateDataSet == null)
            {
                log.debug("Generating sorted dataset for initial cleanup");
                IDataSet unsortedTruncateDataSet = connection.createDataSet();

                ITableFilter filter = new DatabaseSequenceFilter(connection, new FilteredDataSet(
                    getTableFilter(),
                    unsortedTruncateDataSet).getTableNames());
                truncateDataSet = new FilteredDataSet(filter, unsortedTruncateDataSet);
                truncateDataSet = new FilteredDataSet(getTableFilter(), truncateDataSet);
                log.debug("Sorted dataset generated");
            }

            IDataSet orderedDataset = dataSet;

            // if a sorted dataset is available, use table sequence for sorting
            if (truncateDataSet != null)
            {
                ITableFilter filter = new SequenceTableFilter(truncateDataSet.getTableNames());
                orderedDataset = new FilteredDataSet(filter, dataSet);
            }

            try
            {
                if (truncateDataSet != null)
                {
                    DatabaseOperation.DELETE_ALL.execute(connection, truncateDataSet);
                }
                if (dataSet != null)
                {
                    InsertIdentityOperation.INSERT.execute(connection, orderedDataset);
                }
            }
            finally
            {
                connection.close();
            }
        }

        // mimic the Spring OpenSessionInViewFilter
        if (mimicSessionFilter())
        {
            Map<String, SessionFactory> sfbeans = ctx.getBeansOfType(SessionFactory.class);
            if (sfbeans.isEmpty())
            {
                fail("No bean of type org.hibernate.SessionFactory found in spring context");
            }
            this.sessionFactory = sfbeans.get(sfbeans.keySet().iterator().next());

            TransactionSynchronizationManager.bindResource(this.getSessionFactory(), new SessionHolder(this
                .getSessionFactory()
                .openSession()));
        }

    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void tearDown() throws Exception
    {
        if (mimicSessionFilter())
        {
            // close open hibernate sessions, mimic the OpenSessionInViewFilter
            if (TransactionSynchronizationManager.hasResource(this.getSessionFactory()))
            {
                TransactionSynchronizationManager.unbindResource(this.getSessionFactory());
            }
        }

        // regenerate db initial state
        String datesetFileName = "/initial-load.xml";
        InputStream testData = getClass().getResourceAsStream(datesetFileName);

        if (testData != null)
        {
            log.debug("Restoring db state");

            IDataSet dataSet = new XmlDataSet(testData);

            DataSource dataSource = (DataSource) ctx.getBean("dataSource");
            IDatabaseConnection connection = new DatabaseConnection(dataSource.getConnection());

            try
            {
                DatabaseOperation.CLEAN_INSERT.execute(connection, dataSet);
            }
            finally
            {
                connection.close();
            }
        }

        super.tearDown();
    }

    /**
     * Returns the full test name.
     * @see junit.framework.TestCase#getName()
     */
    @Override
    public String getName()
    {
        return ClassUtils.getShortClassName(this.getClass()) + "::" + super.getName();
    }

    /**
     * @return The IDatabase connection to use to insert data
     * @throws SQLException Thrown on any database connection error
     */
    protected IDatabaseConnection createConnection() throws SQLException
    {
        DataSource dataSource = (DataSource) ctx.getBean("dataSource");
        return new DatabaseConnection(dataSource.getConnection());
    }

    /**
     * return the current Hibernate SessionFactory
     * @return SessionFactory object
     */
    protected SessionFactory getSessionFactory()
    {
        return sessionFactory;
    }
}