/*
 * Decompiled with CFR 0.152.
 */
package net.sourceforge.pmd.docs;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;

public class DeadLinksChecker {
    private static final Logger LOG = Logger.getLogger(DeadLinksChecker.class.getName());
    private static final String CHECK_EXTERNAL_LINKS_PROPERTY = "pmd.doc.checkExternalLinks";
    private static final boolean CHECK_EXTERNAL_LINKS = Boolean.parseBoolean(System.getProperty("pmd.doc.checkExternalLinks"));
    private static final Pattern LOCAL_LINK_PATTERN = Pattern.compile("(!)?\\[.*?]\\((.*?)\\)");
    private static final Pattern MD_HEADER_PERMALINK = Pattern.compile("permalink:\\s*(.*)");
    private static final Pattern MD_CAPTION = Pattern.compile("^##+\\s+(.*)$", 8);
    private static final Pattern EXCLUDED_LINK_TARGETS = Pattern.compile("^pmd_userdocs_cli_reference\\.html.*");
    private static final String LOCAL_FILE_PREFIX = "https://github.com/pmd/pmd/blob/master/";
    private static final List<String> IGNORED_URL_PREFIXES = Collections.unmodifiableList(Arrays.asList("https://github.com/pmd/pmd/issues/", "https://github.com/pmd/pmd/pull/", "https://sourceforge.net/p/pmd/bugs/"));
    private final Map<String, CompletableFuture<Integer>> urlResponseCache = new ConcurrentHashMap<String, CompletableFuture<Integer>>();
    private final ExecutorService executorService = Executors.newCachedThreadPool();

    public void checkDeadLinks(Path rootDirectory) {
        Map<Path, List<String>> joined;
        Path pagesDirectory = rootDirectory.resolve("docs/pages");
        Path docsDirectory = rootDirectory.resolve("docs");
        if (!Files.isDirectory(pagesDirectory, new LinkOption[0])) {
            LOG.severe("can't check for dead links, didn't find \"pages\" directory at: " + pagesDirectory);
            System.exit(1);
        }
        List<Path> mdFiles = this.listMdFiles(pagesDirectory);
        HashMap<Path, List<Future<String>>> fileToDeadLinks = new HashMap<Path, List<Future<String>>>();
        Set<String> htmlPages = this.extractLinkTargets(mdFiles);
        int scannedFiles = 0;
        int foundExternalLinks = 0;
        int checkedExternalLinks = 0;
        for (Path mdFile : mdFiles) {
            String pageContent = this.fileToString(mdFile);
            ++scannedFiles;
            String[] lines = pageContent.split("\r?\n");
            for (int index = 0; index < lines.length; ++index) {
                String line = lines[index];
                int lineNo = index + 1;
                Matcher matcher = LOCAL_LINK_PATTERN.matcher(line);
                block2: while (matcher.find()) {
                    boolean linkOk;
                    String linkText = matcher.group();
                    boolean isImageLink = matcher.group(1) != null;
                    String linkTarget = matcher.group(2);
                    if (linkTarget.charAt(0) == '/') {
                        linkOk = false;
                    } else if (linkTarget.startsWith(LOCAL_FILE_PREFIX)) {
                        Path localFile;
                        Object localLinkPart = linkTarget.substring(LOCAL_FILE_PREFIX.length());
                        if (((String)localLinkPart).contains("#")) {
                            localLinkPart = ((String)localLinkPart).substring(0, ((String)localLinkPart).indexOf(35));
                        }
                        if (!(linkOk = Files.isRegularFile(localFile = rootDirectory.resolve((String)localLinkPart), new LinkOption[0]))) {
                            LOG.warning("local file not found: " + localFile);
                            LOG.warning("  linked by: " + linkTarget);
                        }
                    } else if (linkTarget.startsWith("http://") || linkTarget.startsWith("https://")) {
                        ++foundExternalLinks;
                        if (!CHECK_EXTERNAL_LINKS) {
                            LOG.finer("ignoring check of external url: " + linkTarget);
                            continue;
                        }
                        for (String ignoredUrlPrefix : IGNORED_URL_PREFIXES) {
                            if (!linkTarget.startsWith(ignoredUrlPrefix)) continue;
                            LOG.finer("not checking link: " + linkTarget);
                            continue block2;
                        }
                        ++checkedExternalLinks;
                        linkOk = true;
                        CompletionStage futureMessage = ((CompletableFuture)this.getCachedFutureResponse(linkTarget).thenApply(c -> c >= 400)).thenApply(dead -> dead != false ? String.format("%8d: %s", lineNo, linkText) : null);
                        this.addDeadLink(fileToDeadLinks, mdFile, (Future<String>)((Object)futureMessage));
                    } else {
                        Path localResource;
                        if (linkTarget.startsWith("#") || EXCLUDED_LINK_TARGETS.matcher(linkTarget).matches()) continue;
                        if (isImageLink) {
                            localResource = docsDirectory.resolve(linkTarget);
                            linkOk = Files.exists(localResource, new LinkOption[0]);
                        } else {
                            boolean bl = linkOk = linkTarget.isEmpty() || htmlPages.contains(linkTarget);
                        }
                        if (!linkOk) {
                            localResource = docsDirectory.resolve(linkTarget);
                            linkOk = Files.exists(localResource, new LinkOption[0]);
                        }
                    }
                    if (linkOk) continue;
                    FutureTask<String> futureTask = new FutureTask<String>(() -> String.format("%8d: %s", lineNo, linkText));
                    futureTask.run();
                    this.addDeadLink(fileToDeadLinks, mdFile, futureTask);
                }
            }
        }
        this.executorService.shutdown();
        LOG.info("Scanned " + scannedFiles + " files for dead links.");
        LOG.info("  Found " + foundExternalLinks + " external links, " + checkedExternalLinks + " of those where checked.");
        if (!CHECK_EXTERNAL_LINKS) {
            LOG.info("External links weren't checked, set -Dpmd.doc.checkExternalLinks=true to enable it.");
        }
        if (!(joined = this.joinFutures(fileToDeadLinks)).isEmpty()) {
            LOG.warning("Found dead link(s):");
            for (Path file : joined.keySet()) {
                System.err.println(rootDirectory.relativize(file).toString());
                joined.get(file).forEach(LOG::warning);
            }
            throw new AssertionError((Object)"Dead links detected");
        }
        LOG.info("No errors found!");
    }

    private Map<Path, List<String>> joinFutures(Map<Path, List<Future<String>>> map) {
        HashMap<Path, List<String>> joined = new HashMap<Path, List<String>>();
        for (Path p : map.keySet()) {
            List evaluatedResult = map.get(p).stream().map(f -> {
                try {
                    return (String)f.get();
                }
                catch (InterruptedException | ExecutionException e) {
                    e.printStackTrace();
                    return null;
                }
            }).filter(Objects::nonNull).sorted(Comparator.naturalOrder()).collect(Collectors.toList());
            if (evaluatedResult.isEmpty()) continue;
            joined.put(p, evaluatedResult);
        }
        return joined;
    }

    private void addDeadLink(Map<Path, List<Future<String>>> fileToDeadLinks, Path file, Future<String> line) {
        fileToDeadLinks.computeIfAbsent(file, k -> new ArrayList()).add(line);
    }

    private Set<String> extractLinkTargets(List<Path> mdFiles) {
        HashSet<String> htmlPages = new HashSet<String>();
        for (Path mdFile : mdFiles) {
            String pageContent = this.fileToString(mdFile);
            Matcher permalinkMatcher = MD_HEADER_PERMALINK.matcher(pageContent);
            if (!permalinkMatcher.find()) continue;
            String pageUrl = permalinkMatcher.group(1).replaceAll("^/+", "");
            htmlPages.add(pageUrl);
            Matcher captionMatcher = MD_CAPTION.matcher(pageContent);
            while (captionMatcher.find()) {
                String anchor = captionMatcher.group(1).toLowerCase(Locale.ROOT).replaceAll("[^a-z0-9_]+", "-").replaceAll("^-+|-+$", "");
                htmlPages.add(pageUrl + "#" + anchor);
            }
        }
        return htmlPages;
    }

    private List<Path> listMdFiles(Path pagesDirectory) {
        try {
            return Files.walk(pagesDirectory, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".md")).collect(Collectors.toList());
        }
        catch (IOException ex) {
            throw new RuntimeException("error listing files in " + pagesDirectory, ex);
        }
    }

    private String fileToString(Path mdFile) {
        String string;
        block8: {
            InputStream inputStream = Files.newInputStream(mdFile, new OpenOption[0]);
            try {
                string = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
                if (inputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ex) {
                    throw new RuntimeException("error reading " + mdFile, ex);
                }
            }
            inputStream.close();
        }
        return string;
    }

    private CompletableFuture<Integer> getCachedFutureResponse(String url) {
        if (this.urlResponseCache.containsKey(url)) {
            LOG.info("response: HTTP " + this.urlResponseCache.get(url) + " (CACHED) on " + url);
            return this.urlResponseCache.get(url);
        }
        CompletableFuture<Integer> futureResponse = CompletableFuture.supplyAsync(() -> this.computeHttpResponse(url), this.executorService);
        this.urlResponseCache.put(url, futureResponse);
        return futureResponse;
    }

    private int computeHttpResponse(String url) {
        try {
            HttpURLConnection httpURLConnection = (HttpURLConnection)new URL(url).openConnection();
            httpURLConnection.setRequestMethod("HEAD");
            httpURLConnection.setConnectTimeout(5000);
            httpURLConnection.setReadTimeout(15000);
            httpURLConnection.connect();
            int responseCode = httpURLConnection.getResponseCode();
            String response = "HTTP " + responseCode;
            if (httpURLConnection.getHeaderField("Location") != null) {
                response = response + ", Location: " + httpURLConnection.getHeaderField("Location");
            }
            LOG.fine("response: " + response + " on " + url);
            return responseCode;
        }
        catch (IOException ex) {
            LOG.fine("response: " + ex.getClass().getName() + " on " + url + " : " + ex.getMessage());
            return 599;
        }
    }

    public static void main(String[] args) throws IOException {
        if (args.length != 1) {
            System.err.println("Wrong arguments!");
            System.err.println();
            System.err.println("java " + DeadLinksChecker.class.getSimpleName() + " <project base directory>");
            System.exit(1);
        }
        Path rootDirectory = Paths.get(args[0], new String[0]).resolve("..").toRealPath(new LinkOption[0]);
        DeadLinksChecker deadLinksChecker = new DeadLinksChecker();
        deadLinksChecker.checkDeadLinks(rootDirectory);
    }
}

