package net.sozal.stackwriter.agent;

import net.sozal.stackwriter.agent.logger.Logger;
import net.sozal.stackwriter.api.ErrorListener;
import net.sozal.stackwriter.api.domain.Frame;

import java.util.*;

/**
 * @author serkan
 */
public final class ExceptionSnapshotSupport {

    static {
        Logger.debug("<ExceptionSnapshotSupport> Loaded by classloader " +
                ExceptionSnapshotSupport.class.getClassLoader());
    }

    private static final ThreadLocal<WeakHashMap<Throwable, Frame[]>> cache =
        new ThreadLocal<WeakHashMap<Throwable, Frame[]>>() {
            @Override
            protected WeakHashMap<Throwable, Frame[]> initialValue() {
                return new WeakHashMap<>();
            }
        };
    private static final ThreadLocal<Boolean> onErrorInProgress = new ThreadLocal<>();

    private static final Set<String> appPackages = new HashSet<>();
    private static final Set<Class<? extends Throwable>> ignoredExceptionTypes =
            new HashSet<Class<? extends Throwable>>() {{
                add(ClassNotFoundException.class);
            }};
    private static ErrorListener errorListener;

    private ExceptionSnapshotSupport() {
    }

    public static boolean shouldTakeSnapshot(Throwable error, int numFrames) {
        if (Boolean.TRUE.equals(onErrorInProgress.get())) {
            return false;
        }
        if (ignoredExceptionTypes.contains(error.getClass())) {
            return false;
        }
        try {
            // Only cache frames when 'in app' packages are provided
            if (appPackages.isEmpty()) {
                return false;
            }

            // Many libraries/frameworks seem to rethrow the same object with trimmed stacktraces,
            // which means later ("smaller") throws would overwrite the existing object in cache.
            // For this reason we prefer the throw with the greatest stack length...
            Map<Throwable, Frame[]> weakMap = cache.get();
            Frame[] existing = weakMap.get(error);
            if (existing != null && numFrames <= existing.length) {
                return false;
            }

            boolean inAppPackages = false;

            // Check each frame against all "in app" package prefixes
            for (StackTraceElement stackTraceElement : error.getStackTrace()) {
                for (String appFrame : appPackages) {
                    if (stackTraceElement.getClassName().startsWith(appFrame)) {
                        inAppPackages = true;
                        break;
                    }
                }
            }

            if (!inAppPackages) {
                return false;
            }

            ErrorListener ec = errorListener;
            if (ec == null) {
                return true;
            } else {
                return ec.shouldTakeSnapshot(error);
            }
        } catch (Throwable t) {
            Logger.error(
                    "<ExceptionSnapshotSupport> " +
                            "Error occurred while checking whether snapshot should be take on error: " + error.getMessage(),
                    t);
            return false;
        }
    }

    public static void onError(Throwable error, Frame[] frames) {
        if (Boolean.TRUE.equals(onErrorInProgress.get())) {
            return;
        }
        try {
            onErrorInProgress.set(true);
            Map<Throwable, Frame[]> weakMap = cache.get();
            weakMap.put(error, frames);
            ErrorListener el = errorListener;
            if (el != null) {
                el.onError(error, frames);
            }
        } catch (Throwable t) {
            Logger.error("<ExceptionSnapshotSupport> Error occurred on error: " + error.getMessage(), t);
        } finally {
            onErrorInProgress.set(false);
        }
    }

    public static void addAppPackage(String appPackage) {
        appPackages.add(appPackage);
    }

    public static void addIgnoredExceptionType(Class<? extends Throwable> exceptionType) {
        Logger.debug(String.format(
                "<ExceptionSnapshotSupport> Adding ignored exception type (in classloader=%s): %s",
                exceptionType.getClassLoader(), exceptionType));
        ignoredExceptionTypes.add(exceptionType);
    }

    public static void setErrorListener(ErrorListener errorListener) {
        Logger.debug(String.format(
                "<ExceptionSnapshotSupport> Setting error listener (in classloader=%s): %s",
                errorListener.getClass().getClassLoader(), errorListener));
        ExceptionSnapshotSupport.errorListener = errorListener;
    }

}
