package net.sozal.stackwriter.agent.logger;

import net.sozal.stackwriter.agent.util.ExceptionUtils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Standard logger implementation which prints to <code>stdout</code> or <code>stderr</code>.
 *
 * @author serkan
 */
public final class Logger {

    private static final String STACKWRITER_PREFIX = "[STACKWRITER] ";
    private static final String TRACE_LEVEL = "TRACE ";
    private static final String DEBUG_LEVEL = "DEBUG ";
    private static final String INFO_LEVEL = "INFO  ";
    private static final String WARN_LEVEL = "WARN  ";
    private static final String ERROR_LEVEL = "ERROR ";
    private static final int TRACE_LEVEL_CODE = 0;
    private static final int DEBUG_LEVEL_CODE = 1;
    private static final int INFO_LEVEL_CODE = 2;
    private static final int WARN_LEVEL_CODE = 3;
    private static final int ERROR_LEVEL_CODE = 4;
    private static final Map<String, Integer> LOG_LEVEL_CODES = new HashMap<String, Integer>() {{
        put(TRACE_LEVEL.trim(), TRACE_LEVEL_CODE);
        put(DEBUG_LEVEL.trim(), DEBUG_LEVEL_CODE);
        put(INFO_LEVEL.trim(), INFO_LEVEL_CODE);
        put(WARN_LEVEL.trim(), WARN_LEVEL_CODE);
        put(ERROR_LEVEL.trim(), ERROR_LEVEL_CODE);
    }};
    private static final String LOG_LEVEL_CONFIG_NAME = "stackwriter.log.level";
    private static final DateFormat TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

    private static final int LOG_LEVEL = getLogLevel();

    private Logger() {
    }

    private static int getLogLevel() {
        for (String propName : System.getProperties().stringPropertyNames()) {
            String sysPropName = propName.trim();
            String sysPropValue = System.getProperty(propName);
            if (LOG_LEVEL_CONFIG_NAME.equalsIgnoreCase(sysPropName)) {
                Integer logLevelCode = LOG_LEVEL_CODES.get(sysPropValue.toUpperCase());
                if (logLevelCode != null) {
                    return logLevelCode;
                }
            }
        }

        for (Map.Entry<String, String> e : System.getenv().entrySet()) {
            String envVarName = e.getKey().trim();
            String envVarValue = e.getValue().trim();
            if (LOG_LEVEL_CONFIG_NAME.equalsIgnoreCase(envVarName.replace("_", "."))) {
                Integer logLevelCode = LOG_LEVEL_CODES.get(envVarValue.toUpperCase());
                if (logLevelCode != null) {
                    return logLevelCode;
                }
            }
        }

        return ERROR_LEVEL_CODE;
    }

    private static String getTime() {
        return TIME_FORMAT.format(new Date());
    }

    private static String getLogPrefix(String level) {
        return STACKWRITER_PREFIX + level + getTime() + " [" + Thread.currentThread().getName() + "] " + ": ";
    }

    public static void trace(String message) {
        if (LOG_LEVEL <= TRACE_LEVEL_CODE) {
            System.out.println(getLogPrefix(TRACE_LEVEL) + message);
        }
    }

    public static void debug(String message) {
        if (LOG_LEVEL <= DEBUG_LEVEL_CODE) {
            System.out.println(getLogPrefix(DEBUG_LEVEL) + message);
        }
    }

    public static void info(String message) {
        if (LOG_LEVEL <= INFO_LEVEL_CODE) {
            System.out.println(getLogPrefix(INFO_LEVEL) + message);
        }
    }

    public static void warn(String message) {
        if (LOG_LEVEL <= WARN_LEVEL_CODE) {
            System.out.println(getLogPrefix(WARN_LEVEL) + message);
        }
    }

    public static void error(String message) {
        if (LOG_LEVEL <= ERROR_LEVEL_CODE) {
            System.err.println(getLogPrefix(ERROR_LEVEL) + message);
        }
    }

    public static void error(String message, Throwable error) {
        if (LOG_LEVEL <= ERROR_LEVEL_CODE) {
            System.err.println(getLogPrefix(ERROR_LEVEL) + message);
            System.err.println(ExceptionUtils.toString(error));
        }
    }

}
